/*
 * @(#)PrefixApplet.java  1.0 97/07/16
 */

import java.lang.*;
import java.awt.*;
import java.io.*;
import java.applet.Applet;

/**
 * Applet for optimization of prefix graphs and generation of VHDL code
 * for prefix adders.
 *
 * @version   1.0, 97/07/16
 * @author    Reto Zimmermann
 * @since     JDK1.0
 */
public class PrefixApplet extends Applet {
    Label label;
    Label depthLabel;
    Label sizeLabel;
    Label depthmaxLabel;
    Label sizemaxLabel;
    Label maxbLabel;
    Label maxlLabel;
    TextField scaleTextField;
    TextField widthTextField;
    TextField heightTextField;
    TextField alldepthTextField;
    TextField allsizeTextField;
    TextField allmaxbTextField;
    TextField allmaxlTextField;
    TextField filenameTextField;
    Button initButton;
    Button compressButton;
    Button expandButton;
    Button shiftupButton;
    Button shiftdownButton;
    Button synthesizeButton;
    Button vhdlButton;
    PrefixCanvas prefixCanvas;
    
    final int WIDTH_MAX = 128, HEIGHT_MAX = 256;
    int scale = 3;
    int width = 16, height = 16, allmaxb = 0, allmaxl = 0;
    int alldepth = height-1, allsize = 0;
    int depth, size, depthmax, sizemax, maxb, maxl;
    boolean fit, transform, recursive;
    
    int[][] node = new int[WIDTH_MAX][HEIGHT_MAX];
    int[] rmin = new int[WIDTH_MAX], rmax = new int[WIDTH_MAX];
    int[] rlen = new int[WIDTH_MAX];
    int[] rnob = new int[WIDTH_MAX], rnol = new int[HEIGHT_MAX];
    int[] ract = new int[WIDTH_MAX];
    
    
    void addComponent(Component comp, int x, int y, 
		      GridBagLayout gridBag, GridBagConstraints c) {
	c.gridx = x;
	c.gridy = y;
	gridBag.setConstraints(comp, c);
	add(comp);
	c.weightx = 1.0;
	c.weighty = 0.0;
	c.gridwidth = c.gridheight = 1;
	c.fill = GridBagConstraints.NONE;
	c.anchor = GridBagConstraints.CENTER;
    }
    
    public void init() {
	GridBagLayout gridBag = new GridBagLayout();
	GridBagConstraints c = new GridBagConstraints();
	
	setLayout(gridBag);
	
	label = new Label("Display");
	c.gridwidth = 2;
	addComponent(label, 0, 0, gridBag, c);
	
	label = new Label("Width:");
	c.anchor = GridBagConstraints.EAST;
	addComponent(label, 0, 1, gridBag, c);
	widthTextField = new TextField(String.valueOf(width), 3);
	c.anchor = GridBagConstraints.WEST;
	addComponent(widthTextField, 1, 1, gridBag, c);
	
	label = new Label("Height:");
	c.anchor = GridBagConstraints.EAST;
	addComponent(label, 0, 2, gridBag, c);
	heightTextField = new TextField(String.valueOf(height-1), 3);
	c.anchor = GridBagConstraints.WEST;
	addComponent(heightTextField, 1, 2, gridBag, c);
	
	label = new Label("Scale:");
	c.anchor = GridBagConstraints.EAST;
	addComponent(label, 0, 3, gridBag, c);
	scaleTextField = new TextField(String.valueOf(scale), 3);
	c.anchor = GridBagConstraints.WEST;
	addComponent(scaleTextField, 1, 3, gridBag, c);
	
	
	label = new Label("Constraints");
	c.gridwidth = 2;
	addComponent(label, 2, 0, gridBag, c);
	
	label = new Label("Depth:");
	c.anchor = GridBagConstraints.EAST;
	addComponent(label, 2, 1, gridBag, c);
	alldepthTextField = new TextField(String.valueOf(alldepth), 3);
	c.anchor = GridBagConstraints.WEST;
	addComponent(alldepthTextField, 3, 1, gridBag, c);
	
	label = new Label("Size:");
	c.anchor = GridBagConstraints.EAST;
	addComponent(label, 2, 2, gridBag, c);
	allsizeTextField = new TextField(String.valueOf(allsize), 3);
	c.anchor = GridBagConstraints.WEST;
	addComponent(allsizeTextField, 3, 2, gridBag, c);
	
	label = new Label("Max o/l:");
	c.anchor = GridBagConstraints.EAST;
	addComponent(label, 2, 3, gridBag, c);
	allmaxlTextField = new TextField(String.valueOf(allmaxl), 3);
	c.anchor = GridBagConstraints.WEST;
	addComponent(allmaxlTextField, 3, 3, gridBag, c);
	
	label = new Label("Max o/b:");
	c.anchor = GridBagConstraints.EAST;
	addComponent(label, 2, 4, gridBag, c);
	allmaxbTextField = new TextField(String.valueOf(allmaxb), 3);
	c.anchor = GridBagConstraints.WEST;
	addComponent(allmaxbTextField, 3, 4, gridBag, c);
	
	
	label = new Label("Functions");
	c.weightx = 0.0;
	c.gridwidth = 3;
	addComponent(label, 4, 0, gridBag, c);
	
	initButton = new Button("Initialize");
	c.weightx = 0.0;
	c.gridheight = 2;
	c.fill = GridBagConstraints.BOTH;
	addComponent(initButton, 4, 1, gridBag, c);
	
	compressButton = new Button("Compress");
	c.weightx = 0.0;
	c.fill = GridBagConstraints.HORIZONTAL;
	addComponent(compressButton, 5, 1, gridBag, c);
	
	expandButton = new Button("Expand");
	c.weightx = 0.0;
	c.fill = GridBagConstraints.HORIZONTAL;
	addComponent(expandButton, 6, 1, gridBag, c);
	
	shiftupButton = new Button("Shift Up");
	c.weightx = 0.0;
	c.fill = GridBagConstraints.HORIZONTAL;
	addComponent(shiftupButton, 5, 2, gridBag, c);
	
	shiftdownButton = new Button("Shift Down");
	c.weightx = 0.0;
	c.fill = GridBagConstraints.HORIZONTAL;
	addComponent(shiftdownButton, 6, 2, gridBag, c);
	
	synthesizeButton = new Button("Synthesize");
	c.gridwidth = 3;
	c.weightx = 0.0;
	c.fill = GridBagConstraints.HORIZONTAL;
	addComponent(synthesizeButton, 4, 3, gridBag, c);
	
	vhdlButton = new Button("VHDL");
	c.weightx = 0.0;
	c.fill = GridBagConstraints.HORIZONTAL;
	addComponent(vhdlButton, 4, 4, gridBag, c);
	filenameTextField = new TextField("/tmp/adder.vhd");
	c.gridwidth = 2;
	c.fill = GridBagConstraints.HORIZONTAL;
	addComponent(filenameTextField, 5, 4, gridBag, c);
	
	
	label = new Label("Results");
	c.gridwidth = 3;
	addComponent(label, 7, 0, gridBag, c);
	
	label = new Label("Depth:");
	c.anchor = GridBagConstraints.EAST;
	addComponent(label, 7, 1, gridBag, c);
	depthLabel = new Label("0	 ");
	c.weightx = 0.0;
	c.anchor = GridBagConstraints.WEST;
	addComponent(depthLabel, 8, 1, gridBag, c);
	depthmaxLabel = new Label("			");
	c.weightx = 0.0;
	c.anchor = GridBagConstraints.WEST;
	addComponent(depthmaxLabel, 9, 1, gridBag, c);
	
	label = new Label("Size:");
	c.anchor = GridBagConstraints.EAST;
	addComponent(label, 7, 2, gridBag, c);
	sizeLabel = new Label("0	 ");
	c.weightx = 0.0;
	c.anchor = GridBagConstraints.WEST;
	addComponent(sizeLabel, 8, 2, gridBag, c);
	sizemaxLabel = new Label("			");
	c.weightx = 0.0;
	c.anchor = GridBagConstraints.WEST;
	addComponent(sizemaxLabel, 9, 2, gridBag, c);
	
	label = new Label("Max o/l:");
	c.anchor = GridBagConstraints.EAST;
	addComponent(label, 7, 3, gridBag, c);
	maxlLabel = new Label("0	 ");
	c.weightx = 0.0;
	c.anchor = GridBagConstraints.WEST;
	addComponent(maxlLabel, 8, 3, gridBag, c);
	
	label = new Label("Max o/b:");
	c.anchor = GridBagConstraints.EAST;
	addComponent(label, 7, 4, gridBag, c);
	maxbLabel = new Label("0	 ");
	c.weightx = 0.0;
	c.anchor = GridBagConstraints.WEST;
	addComponent(maxbLabel, 8, 4, gridBag, c);
	
	
	prefixCanvas = new PrefixCanvas(this);
	c.weightx = 1.0;
	c.weighty = 1.0;
	c.fill = GridBagConstraints.BOTH;
	c.gridwidth = GridBagConstraints.REMAINDER;
	c.gridheight = GridBagConstraints.REMAINDER;
	prefixCanvas.setBackground(Color.white);
	addComponent(prefixCanvas, 0, 5, gridBag, c);
	
	initGraph();
	validate();
    }
    
    public boolean action(Event evt, Object arg) {
	if (evt.target == scaleTextField) {
	    scale = Integer.parseInt(scaleTextField.getText());
	    prefixCanvas.initSize(width, height);
	    prefixCanvas.repaint();
	} else if (evt.target == widthTextField) {
	    width = Integer.parseInt(widthTextField.getText());
	    initGraph();
	    prefixCanvas.repaint();
	} else if (evt.target == heightTextField) {
	    height = Integer.parseInt(heightTextField.getText()) + 1;
	    initGraph();
	    prefixCanvas.repaint();
	} else if (evt.target == alldepthTextField) {
	    int l = alldepth;
	    alldepth = Integer.parseInt(alldepthTextField.getText());
	    l = alldepth - l;
	    for (int i = 0; i < width; i++) rmax[i] += l;
	    prefixCanvas.repaint();
	} else if (evt.target == allsizeTextField) {
	    allsize = Integer.parseInt(allsizeTextField.getText());
	} else if (evt.target == allmaxbTextField) {
	    allmaxb = Integer.parseInt(allmaxbTextField.getText());
	} else if (evt.target == allmaxlTextField) {
	    allmaxl = Integer.parseInt(allmaxlTextField.getText());
	} else if (evt.target == initButton) {
	    serialGraph();
	    prefixCanvas.repaint();
	} else if (evt.target == compressButton) {
	    transform = true;
	    compressGraph();
	    prefixCanvas.repaint();
	} else if (evt.target == expandButton) {
	    transform = true;
	    expandGraph();
	    prefixCanvas.repaint();
	} else if (evt.target == shiftupButton) {
	    transform = false;
	    compressGraph();
	    prefixCanvas.repaint();
	} else if (evt.target == shiftdownButton) {
	    transform = false;
	    expandGraph();
	    prefixCanvas.repaint();
	} else if (evt.target == synthesizeButton) {
	    serialGraph();
	    transform = true;
	    compressGraph();
	    expandGraph();
	    transform = false;
	    compressGraph();
	    prefixCanvas.repaint();
	} else if (evt.target == vhdlButton) {
	    outputVHDL();
	}
	return false;
    }
    
    
    void cleanGraph() {
	int i, l = HEIGHT_MAX;
	for (i = 0; i < width; i++) l = Math.min(l, rmin[i]);
	for (i = 0; i < width; i++) { rmin[i] -= l; rmax[i] -= l; }
	l = 0;
	for (i = 0; i < width; i++) l = (l > rmax[i]) ? l : rmax[i];
	alldepth = l; allsize = 0;
	alldepthTextField.setText(Integer.toString(alldepth));
	allsizeTextField.setText(Integer.toString(allsize));
    }
    
    void initNodes() {
	for (int i = 0; i < width; i++) {
	    for (int l = 0; l < 2*height; l++) node[i][l] = -1;
	}
	for (int l = 0; l < 2*height; l++) rnol[l] = 0;
    }
    
    void initGraph() {
	initNodes();
	for (int i = 0; i < width; i++) { 
	    rmin[i] = 0; rmax[i] = height-1; rnob[i] = 0;
	}
	for (int l = 0; l < 2*height; l++) rnol[l] = 0;
	prefixCanvas.initSize(width, height);
	depth = size = depthmax = sizemax = maxb = allsize = 0;
	alldepth = height-1;
	alldepthTextField.setText(Integer.toString(alldepth));
	allsizeTextField.setText("0");
	depthLabel.setText("0	 ");
	sizeLabel.setText("0	 ");
	depthmaxLabel.setText("		 ");
	sizemaxLabel.setText("		 ");
	maxbLabel.setText("0	 ");
	maxlLabel.setText("0	 ");
    }
    
    void serialGraph() {
	initNodes();
	for (int i = 1, l = rmin[0]; i < width; i++) {
	    l = Math.max(rmin[i], l) +1;
	    node[i][l] = i-1;
	    rlen[i] = l;
	    rnob[i] = rnol[l] = 1;
	    for (int k = l-1; k >= 0; k--) node[i][k] = -1;
	}
	fixup(false);
    }
    
    void fixup(boolean opt) {
	int l;
	size = 0;
	depth = 1;
	maxb = maxl = 0;
	fit = true;
	for (int i = 0; i < width; i++) {
	    for (l = Math.max(rlen[i], rmax[i]); node[i][l] == -1 && l > 0;
		 l--);
	    rlen[i] = l;
	    depth = Math.max(depth, rlen[i]);
	    for (l = 0; l <= rlen[i]; l++) 
		if (node[i][l] != -1) size++;
	    maxb = Math.max(maxb, rnob[i]);
	    if (rlen[i] > rmax[i]) fit = false;
	}
	for (l = 0; l <= depth; l++) maxl = Math.max(maxl, rnol[l]);
	if (opt) { depthmax = depth; sizemax = size; }
	depthLabel.setText(Integer.toString(depth));
	sizeLabel.setText(Integer.toString(size));
	if (!fit) depthmaxLabel.setText("(*)");
	else if (depthmax != 0) 
	    depthmaxLabel.setText("(" + Integer.toString(depthmax) + ")");
	else depthmaxLabel.setText("		 ");
	if (!fit) sizemaxLabel.setText("(*)");
	else if (sizemax != 0) 
	    sizemaxLabel.setText("(" + Integer.toString(sizemax) + ")");
	else sizemaxLabel.setText("			");
	maxbLabel.setText(Integer.toString(maxb));
	maxlLabel.setText(Integer.toString(maxl));
    }
    
    // return value = free location for black node at (i, l)
    boolean compressColumn(int i, int l, boolean first) {
	int i1, i2;
	if (!recursive && !first) return false;
	// too many black nodes in column i -> stop recursion
	else if (rnob[i] > allmaxb && allmaxb != 0) return false;
	// too many black nodes in row l -> stop recursion
	else if (rnol[l] > allmaxl && allmaxl != 0) return false;
	// top of column i -> stop recursion
	else if (l <= rmin[i]+1) return (node[i][l] == -1);
	// white node (i, l) -> node (i, l-1)
	else if ((i1 = node[i][l]) == -1) {
	    compressColumn(i, l-1, false);
	    return true;
	    // single black node (i, l)
	} else if ((i2 = node[i1][l-1]) == -1) {
	    // top of column i1 -> node (i, l-1)
	    if (l-1 <= rmin[i1]) {
		compressColumn(i, l-1, false);
		return false;
	    }
	    // black node (i, l-1) -> new recursion for node (i, l-1)
	    else if (node[i][l-1] != -1 && !compressColumn(i, l-1, false)) {
		return false;
		// -> shift up
	    } else if (rnol[l-1] < allmaxl || allmaxl == 0) {
		node[i][l-1] = i1;
		node[i][l] = -1;
		rnol[l]--;
		rnol[l-1]++;
		compressColumn(i, l-1, false);
		return true;
	    } else return false;
	    // -> series of two black nodes (i, l) and (i1, l-1)
	} else if (transform) {
	    // black node (i, l-1) -> new recursion for node (i, l-1)
	    if (node[i][l-1] != -1 && !compressColumn(i, l-1, false)) 
		return false;
	    else {
		node[i][l-1] = i1;
		rnob[i]++;
		rnol[l-1]++;
		// shift up black node (i, l-1) possible
		// -> factorize, node (i, l-1)
		if (compressColumn(i, l-1, true)) {
		    node[i][l-1] = i2;
		    rnol[l]--;
		    rnol[l-1]++;
		    node[i][l] = -1;
		    return true;
		    // not factorize
		} else {
		    node[i][l-1] = -1;
		    rnob[i]--;
		    rnol[l-1]--;
		    return false;
		}
	    }
	} else {
	    compressColumn(i, l-1, false);
	    return false;
	}
    }
    
    void compressGraph() {
	recursive = true;
	for (int i = 0; i < width; i++) compressColumn(i, rlen[i], true);
	fixup(transform);
    }
    
    // return value = free location for black node at (i, l)
    boolean expandColumn(int i, int l, boolean first) {
	int s, i1, i2;
	if (!recursive && !first) return false;
	// bottom of column i -> stop recursion
	else if (l >= rmax[i]) return false;
	// look for successors of node (i, l)
	for (s = i+1; s < width && node[s][l+1] != i; s++);
	// white node (i, l) -> node (i, l+1)
	if (node[i][l] == -1) {
	    expandColumn(i, l+1, false);
	    return true;
	    // black node (i, l) with successors -> node (i, l+1)
	} else if (s < width) {
	    expandColumn(i, l+1, false);
	    return false;
	    // white node (i, l+1) -> shift down
	} else if (node[i][l+1] == -1) {
	    node[i][l+1] = node[i][l];
	    node[i][l] = -1;
	    rnol[l]--;
	    rnol[l+1]++;
	    expandColumn(i, l+1, false);
	    return true;
	    // factorized black node (i, l)
	} else if (node[node[i][l]][l+1] == node[i][l+1] && transform &&
		   (allsize == 0 || size > allsize)) {
	    i1 = node[i][l];
	    i2 = node[i][l+1];
	    node[i][l+1] = i1;
	    rnol[l+1]++;
	    // shift down black node (i, l+1) possible
	    // -> unfactorize node (i, l+1)
	    if (expandColumn(i, l+1, true)) {
		node[i][l] = -1;
		rnob[i]--;
		size--;
		rnol[l]--;
		rnol[l+1]--;
		return true;
		// not unfactorize
	    } else {
		node[i][l+1] = i2;
		rnol[l+1]--;
		return false;
	    }
	    // not factorized black node (i, l) -> node (i, l+1)
	    // only required for second expansion run
	} else {
	    expandColumn(i, l+1, false);
	    return false;
	}
    }
    
    void expandGraph() {
	if (allmaxl != 0) return;
	recursive = true;
	size = sizemax;
	for (int i = width-1; i >= 0; i--) expandColumn(i, rmin[i], true);
	fixup(false);
    }
    
    void outputVHDL() {
	try {
	    int ib;
	    PrintStream os = new PrintStream(new FileOutputStream(filenameTextField.getText()));
	    os.println("library IEEE;");
	    os.println("use IEEE.STD_LOGIC_1164.ALL;");
	    os.println("");
	    os.println("entity adder is");
	    os.println("		port (A,B : in Std_Logic_Vector(" + (width-1) + " downto 0);");
	    os.println("					CI : in Std_Logic;");
	    os.println("					S : out Std_Logic_Vector(" + (width-1) + " downto 0);");
	    os.println("					CO : out Std_Logic);");
	    os.println("end adder;");
	    os.println("");
	    os.println("architecture structural of adder is");
	    os.println("");
	    os.println("		signal G,P : Std_Logic_Vector(" + (size+width-1) + " downto 0);");
	    os.println("		signal C : Std_Logic_Vector(" + (width-1) + " downto 0);");
	    os.println("");
	    os.println("		procedure square0_node ");
	    os.println("				(signal G,P : out Std_Logic;");
	    os.println("				 signal A,B,CI : in Std_Logic) is");
	    os.println("		begin");
	    os.println("				G <= (A and B) or (A and CI) or (B and CI);");
	    os.println("				P <= A xor B;");
	    os.println("		end square0_node;");
	    os.println("");
	    os.println("		procedure square_node ");
	    os.println("				(signal G,P : out Std_Logic;");
	    os.println("				 signal A,B : in Std_Logic) is");
	    os.println("		begin");
	    os.println("				G <= A and B;");
	    os.println("				P <= A xor B;");
	    os.println("		end square_node;");
	    os.println("");
	    //			os.println("		procedure white_node ");
	    //			os.println("				(signal GO,PO : out Std_Logic;");
	    //			os.println("				 signal GI,PI : in Std_Logic) is");
	    //			os.println("		begin");
	    //			os.println("				GO <= GI; ");
	    //			os.println("				PO <= PI;");
	    //			os.println("		end white_node;");
	    //			os.println("");
	    os.println("		procedure black_node ");
	    os.println("				(signal GO,PO : out Std_Logic;");
	    os.println("				 signal GI1,PI1,GI2,PI2 : in Std_Logic) is");
	    os.println("		begin");
	    os.println("				GO <= GI1 or (PI1 and GI2); ");
	    os.println("				PO <= PI1 and PI2;");
	    os.println("		end black_node;");
	    os.println("");
	    os.println("		procedure diamond_node ");
	    os.println("				(signal S : out Std_Logic;");
	    os.println("				 signal C,P : in Std_Logic) is");
	    os.println("		begin");
	    os.println("				S <= P xor C;");
	    os.println("		end diamond_node;");
	    os.println("");
	    os.println("begin");
	    os.println("");
	    os.println("		square0_node (G(0),P(0),A(0),B(0),CI);");
	    os.println("		preprocessing : for i in 1 to " + (width-1) + " generate");
	    os.println("				square_node (G(i),P(i),A(i),B(i));");
	    os.println("		end generate preprocessing;");
	    os.println("");
	    for (ib = 0; ib < width; ib++) ract[ib] = ib;		
	    for (int l = 1; l <= depth; l++) {
		for (int i = 0, k; i < width; i++) {
		    if ((k = node[i][l]) != -1) {
			os.println("		black_node (" +
				   "G(" + ib + "),P(" + ib + 
				   "),G(" + ract[i] + "),P(" + ract[i] + 
				   "),G(" + ract[k] + "),P(" + ract[k] + "));");
			ract[i] = ib++;
		    }
		    //					else
		    //						os.println("		white_node (G(" + (l*width+i) + "),P(" + (l*width+i) + "),G(" + ((l-1)*width+i) + "),P(" + ((l-1)*width+i) + "));");
		}
	    }
	    os.println("");
	    os.println("		C(0) <= CI;");
	    for (int i = 0; i < width-1; i++)
		os.println("		C(" + (i+1) + ") <= G(" + ract[i] + ");");
	    //			os.println("		carry_generation : for i in 0 to " + (width-2) + " generate");
	    //			os.println("				C(i+1) <= G(" + ((depth)*width) + "+i);");
	    //			os.println("		end generate carry_generation;");
	    os.println("		CO <= G(" + ract[width-1] + ");");
	    os.println("		");
	    os.println("		postprocessing : for i in 0 to " + (width-1) + " generate");
	    os.println("				diamond_node (S(i),C(i),P(i));");
	    os.println("		end generate postprocessing;");
	    os.println("				");
	    os.println("end structural;");
	    os.close();
	} catch (FileNotFoundException e) {
	    System.err.println("FileStreamsTest: " + e);
	} catch (IOException e) {
	    System.err.println("FileStreamsTest: " + e);
	}
    }
    
    public String getAppletInfo() {
        return "PrefixApplet by Reto Zimmermann";
    }
}


class PrefixCanvas extends Canvas {
    PrefixApplet prefixApplet;
    
    final int L_MARG = 10, T_MARG = 40, MINDRAG = 2;
    int DIS, DIAM, RAD; 
    int xOrig, yOrig, xMin, xMax, yMin, yMax, xold, yold, xdown, ydown;
    
    public PrefixCanvas(PrefixApplet prefixApplet) {
	super();
	this.prefixApplet = prefixApplet;
    }
    
    public void initSize(int width, int height) {
	DIS = 4*prefixApplet.scale;
	DIAM = 2*prefixApplet.scale;
	RAD = DIAM/2; 
	xOrig = L_MARG + (width-1) * DIS;
	yOrig = T_MARG+DIS-DIAM;
	xMin = L_MARG; yMin = T_MARG;
	xMax = xOrig + DIAM; yMax = yOrig + (height) * DIS;
    }
    
    public boolean mouseDown(Event event, int x, int y) {
	if (x > xMax) x = xMax; else if (x < xMin) x = xMin;
	if (y > yMax) y = yMax; else if (y < yMin) y = yMin;
	xdown = xold = x;
	ydown = yold = y;
	return false;
    }
    
    public boolean mouseUp(Event event, int x, int y) {
	if (x > xMax) x = xMax; else if (x < xMin) x = xMin;
	if (y > yMax) y = yMax; else if (y < yMin) y = yMin;
	if (x == xdown && y == ydown) {
	    int i = (xOrig-x+DIS-RAD)/DIS;
	    int l = (y-yOrig+RAD)/DIS;
	    prefixApplet.transform = !event.metaDown();
	    if (y < yOrig) {
		prefixApplet.recursive = true;
		if (event.shiftDown()) {
		    prefixApplet.expandColumn(i, 1, true);
		} else {
		    prefixApplet.compressColumn(i, prefixApplet.rlen[i], true);
		}
	    } else {
		prefixApplet.recursive = false;
		if (event.shiftDown()) {
		    prefixApplet.expandColumn(i, l, true);
		} else {
		    prefixApplet.compressColumn(i, l, true);
		}
	    }
	    prefixApplet.fixup(false);
	}
	repaint();
	return false;
    }
    
    public boolean mouseDrag(Event event, int x, int y) {
	if (x > xMax) x = xMax; else if (x < xMin) x = xMin;
	if (y > yMax) y = yMax; else if (y < yMin) y = yMin;
	int i1 = (xOrig-x+RAD+DIS)/DIS;
	int i2 = (xOrig-xold+RAD+DIS)/DIS;
	if (x - xdown > MINDRAG || x - xdown < -MINDRAG) {
	    if (x > xold) {
		for (int i = i1; i < i2; i++) {
		    prefixApplet.rmin[i] = 
			(y+(y-yold)*((i1-i)*DIS)/(x-xold)-yOrig-RAD+DIS)/DIS;
		}
	    } else {
		for (int i = i1-1; i > i2-1; i--) {
		    prefixApplet.rmax[i] = 
			(y+(y-yold)*((i1-i)*DIS)/(x-xold)-yOrig-RAD)/DIS;
		}
	    }
	}
	prefixApplet.cleanGraph();
	xold = x;
	yold = y;
	return false;
    }
    
    public void paint(Graphics g) {
	int i, l, k, j;
	for (i = 0; i < prefixApplet.width; i++) {
	    if (i >= 100)
		g.drawString(String.valueOf(i/100), 
			     xOrig-i*DIS+DIAM/2-2, yOrig-25-DIS+DIAM);
	    if (i >= 10) g.drawString(String.valueOf(i/10%10), 
				      xOrig-i*DIS+DIAM/2-2, yOrig-15-DIS+DIAM);
	    g.drawString(String.valueOf(i%10), 
			 xOrig-i*DIS+DIAM/2-2, yOrig-5-DIS+DIAM);
	    g.drawLine(xOrig-i*DIS+RAD, yOrig-DIS+DIAM, xOrig-i*DIS+RAD, 
		       yOrig+(prefixApplet.height)*DIS);
	}
	for (l = 0; l < prefixApplet.height; l++) {
	    g.drawString(String.valueOf(l), xOrig+DIAM+5,
			 yOrig+l*DIS+DIAM/2+4);
	}
	
	for (l = prefixApplet.height-1; l >= 0; l--) {
	    j = prefixApplet.width;
	    for (i = prefixApplet.width-1; i >= 0; i--) {
		if (i == j) {
		    g.drawLine(xOrig-i*DIS-RAD, yOrig+l*DIS-RAD, 
			       xOrig-i*DIS-RAD+DIS/2, yOrig+l*DIS-RAD-DIS/2);
		} else if (i > j) {
		    g.drawLine(xOrig-i*DIS-RAD, yOrig+l*DIS-RAD, 
			       xOrig-i*DIS+DIS-RAD, yOrig+l*DIS-RAD);
		}
		if ((k = prefixApplet.node[i][l]) >= 0) {
		    g.drawLine(xOrig-i*DIS+RAD, yOrig+l*DIS+RAD, 
			       xOrig-i*DIS+RAD+DIS/2, yOrig+l*DIS+RAD-DIS/2);
		    if (k <= j) { j = k; }
		    else {
			g.drawLine(xOrig-k*DIS-RAD, yOrig+l*DIS-RAD,
				   xOrig-k*DIS-RAD+DIS/2,
				   yOrig+l*DIS-RAD-DIS/2);
		    }
		}
		g.setColor(Color.white);
		g.fillOval(xOrig-i*DIS, yOrig+l*DIS, DIAM, DIAM);
		if (l < prefixApplet.rmin[i]) { g.setColor(Color.green); }
		else if (l > prefixApplet.rmax[i]) { g.setColor(Color.red); }
		else { g.setColor(Color.black); }
		if (prefixApplet.node[i][l] < 0) {
		    g.drawOval(xOrig-i*DIS, yOrig+l*DIS, DIAM, DIAM);
		    g.setColor(Color.black);
		} else {
		    g.fillOval(xOrig-i*DIS, yOrig+l*DIS, DIAM, DIAM);
		    g.setColor(Color.black);
		}
	    }
	}
    }
}
