/*
 *  vvv.y - bison grammar for parsing vhdl transition conditions
 *          and output assignments only.
 *
 *  Tom Mayo - 7-4-95
 */

%{

#include "b20.h"

%}

%union {
	int ival;
	char sval[128];
}

/*
 *  Note:  the "FOO" token is what the lexer sends
 *         for "<=", because it has no idea whether
 *         this is a comparison or assignment.  The
 *         grammar is context-sensitive for this.
 */

%token <sval> LITERAL IDENTIFIER
%token <ival> FOO AND OR NAND NOR XOR XNOR NOT_EQUAL
%token <ival> GREATER_OR_EQUAL MOD REM ABS NOT
%token <ival> EXPONENT_OPERATOR EOL STATE_START TRANS_START ASYNC_START
%token <ival> NUMBER

%left AND OR XOR XNOR
%nonassoc NAND NOR
%left '=' NOT_EQUAL '<' FOO '>' GREATER_OR_EQUAL
%left '+' '-' '&'
%nonassoc SIGN
%left '*' '/' MOD REM
%left EXPONENT_OPERATOR ABS NOT


%%

line: state_or_trans
		{ ; }
	| line state_or_trans
		{ ; }
;

state_or_trans:	STATE_START multi_assignment
		{ ; }
	| TRANS_START expression_and_or_multi
		{ ; }
	| ASYNC_START expression '|' multi_assignment
		{ ; }
	| ASYNC_START expression
		{ ; }
;

expression_and_or_multi: priority expression '|' multi_assignment
		{ ; }
	| priority expression
		{ ; }
	| priority '|' multi_assignment
		{ ; }
	| expression '|' multi_assignment
		{ ; }
	| expression
		{ ; }
	| '|' multi_assignment
		{ ; }
;

multi_assignment:	assignment
		{ ; }
	| multi_assignment assignment;

assignment:	IDENTIFIER FOO expression ';'
		{ add_output($1, outputs_async); }
;

expression:	nand_nor_relation
		{ ; }
	| and_or_xor_xnor_relation
		{ ; }
;

priority:	NUMBER ':'
		{ set_priority($1); }
;

/*
 *  NAND and NOR are non-associative, so
 *  only allow one of these (unless, of
 *  course, parentheses are given).
 */

nand_nor_relation:	relation NAND relation
		{ ; }
	| relation NOR relation
		{ ; }
;

/*
 *  Allow as many of these as you want.
 */

and_or_xor_xnor_relation: and_or_xor_xnor_relation AND relation 
		{ ; }
	| and_or_xor_xnor_relation OR relation 
		{ ; }
	| and_or_xor_xnor_relation XOR relation 
		{ ; }
	| and_or_xor_xnor_relation XNOR relation
		{ ; }
	| relation
		{ ; }
;

relation:	simple_expression
		{ ; }
	|	simple_expression relational_operator simple_expression
		{ ; }

simple_expression:	term
		{ ; }
	|	signed_expression
		{ ; }
	|	signed_expression adding_operator term
		{ ; }
	|	term adding_operator term
		{ ; }
;

signed_expression:	sign term %prec SIGN
		{ ; }
;

term:	factor
		{ ; }
	|	factor multiplying_operator factor
		{ ; }
;

factor:	primary
		{ ; }
	|	primary EXPONENT_OPERATOR primary
		{ ; }
	|	ABS primary
		{ ; }
	|	NOT primary
		{ ; }
;

primary:	IDENTIFIER
		{ add_input($1, inputs_async); }
	|	LITERAL
		{ ; }
	|	'(' expression ')'
		{ ; }
;

/*
 *  Ideally would like to support the following
 *  but unlikely that it will be missed.
 */

/*
primary:	name
	|	literal
	|	qualified_expression
	|	type_conversion
	|	allocator
	|	( expression )
;
*/

relational_operator:	'='
		{ ; }
	|	'<'
		{ ; }
	|	'>'
		{ ; }
	|	NOT_EQUAL
		{ ; }
	|	FOO
		{ ; }
	|	GREATER_OR_EQUAL
		{ ; }
;

adding_operator:	'+'
		{ ; }
	|	'-'
		{ ; }
	|	'&'
		{ ; }
;

sign:	'+'
		{ ; }
	|	'-'
		{ ; }
;

multiplying_operator:	'*'
		{ ; }
	|	'/'
		{ ; }
	|	MOD
		{ ; }
	|	REM
		{ ; }
;

%%

/*
 *  b20iofind - function to sift through all states
 *    and all transitions to pluck out input and output
 *    signals.
 */

void b20iofind(void)
{
	extern int mylineno;
	mylineno = 1;

/*
 *  Go through all states.
 */
 	current_state = first_state;

	while(current_state != (state_s_t *)NULL)
	{
		current_string = current_state->first_output;

		if (debug_iofind)
			fprintf(stderr,
				"-- Parsing strings for state \"%s\".\n",
					current_state->name->text);

		while(current_string != (string_s_t *)NULL)
		{
			strcpy(vvv_lex_string, " state: ");
			strncat(vvv_lex_string, current_string->text,
				LARGE_STRING - 9);
			vvv_lex_end = vvv_lex_string + strlen(vvv_lex_string);
			vvv_lex_ptr = vvv_lex_string;

			if (debug_iofind)
				fprintf(stderr,
					"-- Parsing string %d: \"%s\".\n",
						mylineno, vvv_lex_string);

			vvvparse();

			current_string = current_string->next_this_state;
			mylineno++;
		}
/*
 *  Go through all transitions for this state.
 */
	 	current_transition = current_state->first_transition;

		while(current_transition != (transition_s_t *)NULL)
		{
			current_string = current_transition->condition;

			if ((current_string != (string_s_t *)NULL) &&
				strlen(current_string->text))
			{
				strcpy(vvv_lex_string, " trans: ");
				strncat(vvv_lex_string, current_string->text,
					LARGE_STRING - 9);
				vvv_lex_end = vvv_lex_string + strlen(vvv_lex_string);
				vvv_lex_ptr = vvv_lex_string;

				if (debug_iofind)
					fprintf(stderr,
						"-- Parsing string %d: \"%s\".\n",
							mylineno, vvv_lex_string);

				prio = -1;

				vvvparse();

				if (prio != -1) {
					current_transition->priority = prio;
					if (debug_iofind)
						fprintf(stderr,
							"-- String %d: priority %d.\n",
								mylineno, current_transition->priority);
				}
			}
			else
			{
				if (debug_iofind)
					fprintf(stderr,
						"-- Not parsing string %d, blank.\n", mylineno);
			}

			current_transition = current_transition->next_this_state;
			mylineno++;
		}

		current_state = current_state->next_overall;
	}

/*
 *  Handle the asynchronous transition (if any).
 */
	if (asynch_transition != (transition_s_t *)NULL)
	{
		current_string = asynch_transition->condition;

		if (current_string == (string_s_t *)NULL)
		{
			fatal_error("--    Asynchronous transition has no condition.");
		}

		strcpy(vvv_lex_string, " async: ");
		strncat(vvv_lex_string, current_string->text,
			LARGE_STRING - 9);
		vvv_lex_end = vvv_lex_string + strlen(vvv_lex_string);
		vvv_lex_ptr = vvv_lex_string;

		if (debug_iofind)
			fprintf(stderr,
				"-- Parsing string %d: \"%s\".\n",
					mylineno, vvv_lex_string);

		vvvparse();
	}

	fflush(stderr);
	fflush(stdout);

	if (first_in == (io_s_t *)NULL)
	{
		printf("-- ERROR: Could not identify any inputs.\n");
		printf("--   This could be due to bad VHDL syntax.\n");

		exit(1);
	}

	if (first_out == (io_s_t *)NULL)
	{
		printf("-- ERROR: Could not identify any outputs.\n");
		printf("--   This could be due to bad VHDL syntax.\n");

		exit(1);
	}

	return;
}

/*
 *  Rudimentary error reporting.  This could be much better.
 */

void vvverror(char *s)
{
	extern char *vvvtext;
	extern int mylineno;

	fprintf(stderr, "-- WARNING: VHDL %s near \"%s\" in string %d:\n",
		s, vvvtext, mylineno);
	fprintf(stderr, "--   %s\n", vvv_lex_string);
	fprintf(stderr, "-- VHDL output is suspect.\n");
	
/*
 *  The following is a bit too goofy.
 */

/*
	fprintf(stderr, "-- Sorry, cannot proceed with invalid VHDL.\n");
	fprintf(stderr, "-- Please consult the VHDL LRM over a beer\n");
	fprintf(stderr, "-- and try again later.\n");
*/

	fflush(stderr);
	fflush(stdout);
}
