--------------------------------------------------------------------------
--  DLX PROCESSOR MODEL SUITE
--  Copyright (C) 1995, Martin Gumm
--  University of Stuttgart / Department of Computer Science / IPVR-ISE
--------------------------------------------------------------------------
--  This program is free software; you can redistribute it and/or modify
--  it under the terms of the GNU General Public License as published by
--  the Free Software Foundation; either version 1, or (at your option)
--  any later version.
--
--  This program is distributed in the hope that it will be useful,
--  but WITHOUT ANY WARRANTY; without even the implied warranty of
--  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--  GNU General Public License for more details.
--------------------------------------------------------------------------
--  Last revision date : November 15 1995
--------------------------------------------------------------------------

--------------------------------------------------------------------------
--  *** SYNOPSYS synthesizable code (ver. 3.2.a) ***
--------------------------------------------------------------------------

--------------------------------------------------------------------------
-- MOORE machine:
--
--            X >______            ________________
--                     \   ______>|output_logic    |_______> Y
--                      \ /       |________________|
--                       \        ________________
--                      / \_____>|next_state_logic|______     
--                     /________>|________________|      |
--                     |              _________          |
--                     |             |state    |         |
--                     |_____________|registers|<________|
--                                   |_________|
--
--------------------------------------------------------------------------

--------------------------------------------------------------------------
--  architecture of the finite state machine
--  (Moore machine / microcoded output style / two separated processes
--  for output_logic and next_state_logic)
--
--  file fsm-moore_2.vhd
--------------------------------------------------------------------------

architecture moore_2 of fsm is
  signal current_state : fsm_states;
  signal next_state    : fsm_states;
begin


  --------------------------------------------------------------------------
  -- State switch
  --------------------------------------------------------------------------
  state_switch : process(phi1, reset)
  begin
    --
    -- switch to next state with rising edge of phi1
    --
    if (phi1'event and phi1 = '1') then
      if reset = '1' then
	current_state <= reset_1;
      else
	current_state <= next_state;
      end if;
    end if;
  end process state_switch;

  --------------------------------------------------------------------------
  -- Next state logic
  -- S' = f (X,S)
  --------------------------------------------------------------------------
  next_state_logic : process (current_state, reset, halt, ready, intrpt, super,
                              priv_ins, alu_ovfl, alu_neg, alu_zero, adr_ls2,
 			      adr_msb, dec_1_in, dec_2_in)
  begin
    case current_state is
      ----------------------------------------------------------------
      when reset_1 =>
	next_state <= reset_2;
      ----------------------------------------------------------------
      when reset_2 =>
	next_state <= fetch;
      ----------------------------------------------------------------
      when fetch =>
	--
	-- check for interrupts and halt
	--
    	if ready = '1' then
	  if halt = '1' then
	    next_state <= halt_state;
	  elsif intrpt = '1' then
            next_state <= intrpt_1;
	  else
	    next_state <= decode_pcincr_ab;
	  end if;
	else
	  next_state <= fetch;
	end if;
      ----------------------------------------------------------------
      when decode_pcincr_ab =>
	--
 	-- check for priviledge violation and address overflow
	-- (take input from opc_decoder)
	--
	if super = '0' and priv_ins = '1' then       -- priv ?
 	  next_state <= priv_viol;       
    	elsif alu_ovfl = '1' then                    -- ovad ?
 	  if super = '0' then
	    next_state <= except_ovad;
	  else
	    next_state <= error_state;
	  end if;  
	else
          next_state <= dec_1_in;
	end if;
      ----------------------------------------------------------------
      when memory =>
        next_state <= dec_2_in;
      ----------------------------------------------------------------
      when load_w_1 =>
	--
	-- check for data address violation and priviledge violation
	--
	if (adr_ls2(1) = '1' or adr_ls2(0) = '1') then  -- dav ?
 	  if super = '0' then
	    next_state <= d_adr_viol;
	  else
	    next_state <= error_state;
	  end if;  
	elsif (adr_msb  = '0' and super = '0') then        -- priv ?
 	  if super = '0' then
	    next_state <= priv_viol;
	  else
	    next_state <= error_state;
	  end if;  
        elsif ready = '1' then                           -- memory ready ?
	  next_state <= load_w_2;
	else
	  next_state <= load_w_1;
	end if;
      ----------------------------------------------------------------
      when load_w_2 =>
	next_state <= write_back;
      ----------------------------------------------------------------
      when load_h_1 =>
	--
	-- check for data address violation and priviledge violation
	--
	if adr_ls2(0) = '1' then                         -- dav ?
 	  if super = '0' then
	    next_state <= d_adr_viol;
	  else
	    next_state <= error_state;
	  end if;  
	elsif (adr_msb  = '0' and super = '0') then        -- priv ?
 	  if super = '0' then
	    next_state <= priv_viol;
	  else
	    next_state <= error_state;
	  end if;  
        elsif ready = '1' then                           -- memory ready ?
	  next_state <= load_h_2;
	else
	  next_state <= load_h_1;
	end if;
      ----------------------------------------------------------------
      when load_h_2 =>
        next_state <= write_back;
      ----------------------------------------------------------------
      when load_b_1 =>
	--
	-- check for priviledge violation
	--
	if (adr_msb  = '0' and super = '0') then       -- priv ?
 	  if super = '0' then
	    next_state <= priv_viol;
	  else
	    next_state <= error_state;
	  end if;  
        elsif ready = '1' then                       -- memory ready ?
	  next_state <= load_b_2;
	else
	  next_state <= load_b_1;
	end if;
      ----------------------------------------------------------------
      when load_b_2 =>
        next_state <= write_back;
      ----------------------------------------------------------------
      when store_w_1 =>
	--
 	-- check for data address or priviledge violation
	--
	if adr_ls2(0) = '1' or adr_ls2(1) = '1' then     -- dav ?
 	  if super = '0' then
	    next_state <= d_adr_viol;
	  else
	    next_state <= error_state;
	  end if;  
	elsif (adr_msb  = '0' and super = '0') then        -- priv ?
 	  if super = '0' then
	    next_state <= priv_viol;
	  else
	    next_state <= error_state;
	  end if;  
	else
	  next_state <= store_w_2;
	end if;
     ----------------------------------------------------------------
      when store_w_2 =>
	--
	-- wait on memory ready
	--
        if ready = '1' then
	  next_state <= fetch;
	else
	  next_state <= store_w_2;
	end if;
      ----------------------------------------------------------------
      when store_h_1 =>
	--
	-- check for data address violation and priviledge violation
	--
	if adr_ls2(0) = '1' then                         -- dav ?
 	  if super = '0' then
	    next_state <= d_adr_viol;
	  else
	    next_state <= error_state;
	  end if;  
	elsif (adr_msb  = '0' and super = '0') then        -- priv ?
 	  if super = '0' then
	    next_state <= priv_viol;
	  else
	    next_state <= error_state;
	  end if;  
	else
	  next_state <= store_h_2;
	end if;
      ----------------------------------------------------------------
      when store_h_2 =>
	--
	-- wait on memory ready
	--
	if ready = '1' then
	  next_state <= fetch;   	
	else
	  next_state <= store_h_2;
	end if;
      ----------------------------------------------------------------
      when store_b_1 =>
	--
 	-- check for priviledge violation
	--
	if (adr_msb  = '0' and super = '0') then       -- priv ?
 	  if super = '0' then
	    next_state <= priv_viol;
	  else
	    next_state <= error_state;
	  end if;  
	else
	  next_state <= store_b_2;
	end if;
      ----------------------------------------------------------------
      when store_b_2 =>
	--
	-- wait on memory ready
	--
	if ready = '1' then
	  next_state <= fetch;   	
	else
	  next_state <= store_b_2;
	end if;  	
      ----------------------------------------------------------------
      when movi2s =>
	next_state <= fetch;   	
      ----------------------------------------------------------------
      when movs2i =>
	next_state <= write_back;
      ----------------------------------------------------------------
      when br_eqz =>
	--
	-- check if equal zero : a - 0 = 0 ?
	--
	if alu_zero = '0' then
	  next_state <= fetch;   	
	else
	  next_state <= branch;
	end if;
      ----------------------------------------------------------------
      when br_nez =>
 	--
	-- check if not equal zero : a - 0 /= 0 ?
	--
	if alu_zero = '1' then
	  next_state <= fetch;   	
	else
	  next_state <= branch;
	end if;
      ----------------------------------------------------------------
      when branch =>
	--
	-- check for iav, and priv
	--
	if (adr_ls2(1) = '1' or adr_ls2(0) = '1') then  -- iav ?
 	  if super = '0' then
	    next_state <= i_adr_viol;
	  else
	    next_state <= error_state;
	  end if; 
	elsif adr_msb  = '0' and super = '0' then            -- priv ?
 	  if super = '0' then
	    next_state <= priv_viol;
	  else
	    next_state <= error_state;
	  end if;  
	else
          next_state <= load_pc;
	end if;   	
      ----------------------------------------------------------------
      when jump =>
	--
 	-- check for iav and priv
	--
	if (adr_ls2(1) = '1' or adr_ls2(0) = '1') then  -- iav ?
 	  if super = '0' then
	    next_state <= i_adr_viol;
	  else
	    next_state <= error_state;
	  end if; 
	elsif adr_msb  = '0' and super = '0' then           -- priv ?
 	  if super = '0' then
	    next_state <= priv_viol;
	  else
	    next_state <= error_state;
	  end if;  
	else
          next_state <= load_pc;
	end if;   	
      ----------------------------------------------------------------
      when jump_lnk_1 =>
        next_state <= jump_lnk_2;
      ----------------------------------------------------------------
      when jump_lnk_2 =>
	--
	-- check for iav and priv
	--
	if (adr_ls2(1) = '1' or adr_ls2(0) = '1') then  -- iav ?
 	  if super = '0' then
	    next_state <= i_adr_viol;
	  else
	    next_state <= error_state;
	  end if; 
	elsif adr_msb  = '0' and super = '0' then        -- priv ?
 	  if super = '0' then
	    next_state <= priv_viol;
	  else
	    next_state <= error_state;
	  end if;  
	else
          next_state <= load_pc;
	end if;   	
      ----------------------------------------------------------------
      when jump_reg =>
	--
	-- check for iav and priv
	--
	if (adr_ls2(1) = '1' or adr_ls2(0) = '1') then   -- iav ?
 	  if super = '0' then
	    next_state <= i_adr_viol;
	  else
	    next_state <= error_state;
	  end if; 
	elsif adr_msb  = '0' and super = '0' then        -- priv ?
 	  if super = '0' then
	    next_state <= priv_viol;
	  else
	    next_state <= error_state;
	  end if;  
	else
          next_state <= load_pc;
	end if;   	
      ----------------------------------------------------------------
      when jump_lnk_reg_1 =>
        next_state <= jump_lnk_reg_2;
      ----------------------------------------------------------------
      when jump_lnk_reg_2 =>
	--
	-- check for iav and priv
	--
	if (adr_ls2(1) = '1' or adr_ls2(0) = '1') then  -- iav ?
 	  if super = '0' then
	    next_state <= i_adr_viol;
	  else
	    next_state <= error_state;
	  end if; 
	elsif adr_msb  = '0' and super = '0' then       -- priv ?
 	  if super = '0' then
	    next_state <= priv_viol;
	  else
	    next_state <= error_state;
	  end if;  
	else
          next_state <= load_pc;
	end if;   	
      ----------------------------------------------------------------
      when trap_1 =>
	next_state <= trap_2;
      ----------------------------------------------------------------
      when trap_2 =>
	--
	--  check for iav
	--
	if (adr_ls2(1) = '1' or adr_ls2(0) = '1') then  -- iav ?
 	  if super = '0' then
	    next_state <= i_adr_viol;
	  else
	    next_state <= error_state;
	  end if; 
	else
          next_state <= trap_3;
	end if;   	
      ----------------------------------------------------------------
      when trap_3 =>
	next_state <= load_pc;
      ----------------------------------------------------------------
      when rfe_1 =>
	next_state <= rfe_2;
      ----------------------------------------------------------------
      when rfe_2 =>
	next_state <= fetch;
      ----------------------------------------------------------------
      when load_pc =>
        next_state <= fetch;
      ----------------------------------------------------------------
      when lhi =>
	next_state <= write_back;
      ----------------------------------------------------------------
      when add_1 =>
	--
	-- check for overflow
	--
        if alu_ovfl = '1' then                    -- ovar ?
	  if super = '0' then
	    next_state <= except_ovar;
	  else
	    next_state <= error_state;
	  end if;
	else
	  next_state <= write_back;
	end if;
      ----------------------------------------------------------------
      when sub_1 =>
	--
	-- check for overflow
	--
        if alu_ovfl = '1' then                    -- ovar ?
	  if super = '0' then
	    next_state <= except_ovar;
	  else
	    next_state <= error_state;
	  end if;
	else
	  next_state <= write_back;
	end if;
     ----------------------------------------------------------------
      when and_1 =>
 	next_state <= write_back;
     ----------------------------------------------------------------
      when or_1 =>
 	next_state <= write_back;
      ----------------------------------------------------------------
      when xor_1 =>
 	next_state <= write_back;
      ----------------------------------------------------------------
      when sll_1 =>
 	next_state <= write_back;
     ----------------------------------------------------------------
      when srl_1 =>
 	next_state <= write_back;
     ----------------------------------------------------------------
      when sra_1 =>
 	next_state <= write_back;
      ----------------------------------------------------------------
      when seq_1 =>
	--
	-- check alu_zero = '1'
	--
 	if alu_zero = '1' then
	  next_state <= set_to_1;
	else
	  next_state <= set_to_0;
	end if;
      ----------------------------------------------------------------
      when sne_1 =>
	--
	-- check alu_zero /= '0'
	--
 	if alu_zero = '0' then
	  next_state <= set_to_1;
	else
	  next_state <= set_to_0;
	end if;
      ----------------------------------------------------------------
      when slt_1 =>
	--
	-- check alu_neg = '1'
	--
 	if alu_neg = '1' then
	  next_state <= set_to_1;
	else
	  next_state <= set_to_0;
	end if;
      ----------------------------------------------------------------
      when sle_1 =>
	--
	-- check alu_neg = '1' or alu_zero = '1'
	--
 	if alu_zero = '1' or alu_neg = '1' then
	  next_state <= set_to_1;
	else
	  next_state <= set_to_0;
	end if;
      ----------------------------------------------------------------
      when sgt_1 =>
	--
	-- check alu_neg = '0' and alu_zero = '0'
	--
 	if alu_neg = '0' and alu_zero = '0' then
	  next_state <= set_to_1;
	else
	  next_state <= set_to_0;
	end if;
      ----------------------------------------------------------------
      when sge_1 =>
	--
	-- check alu_neg = '0'
	--
 	if alu_neg = '0' then
	  next_state <= set_to_1;
	else
	  next_state <= set_to_0;
	end if;
      ----------------------------------------------------------------
      when set_to_1 =>
	next_state <= write_back;
      ----------------------------------------------------------------
      when set_to_0 =>
	next_state <= write_back;
      ----------------------------------------------------------------
      when write_back =>
	next_state <= fetch;
      ----------------------------------------------------------------
      when intrpt_1 =>
	next_state <= intrpt_2;
      ----------------------------------------------------------------
      when intrpt_2 =>
	next_state <= intrpt_3;
      ----------------------------------------------------------------
      when intrpt_3 =>
	next_state <= fetch;
      ----------------------------------------------------------------
      when ill_opc =>
	next_state <= intrpt_2;
      ----------------------------------------------------------------
      when ill_rraf =>
	next_state <= intrpt_2;
      ----------------------------------------------------------------
      when d_adr_viol =>
	next_state <= intrpt_2;
      ----------------------------------------------------------------
      when i_adr_viol =>
	next_state <= intrpt_2;
      ----------------------------------------------------------------
      when priv_viol =>
	next_state <= intrpt_2;
      ----------------------------------------------------------------
      when except_ovad =>
	next_state <= intrpt_2;
      ----------------------------------------------------------------
      when except_ovar =>
	next_state <= intrpt_2;
      ----------------------------------------------------------------
      when halt_state =>
	--
	-- wait until halt = '0'
	--
    	if halt = '0' then
	  next_state <= decode_pcincr_ab;
	else
	  next_state <= halt_state;
	end if;
      ----------------------------------------------------------------
      when error_state =>
	--
	-- wait until reset = '1'
	--
    	if reset = '0' then
	  next_state <= error_state;
	else
	  next_state <= reset_1;
	end if;
      ----------------------------------------------------------------
     end case;
  end process next_state_logic;
  
  ----------------------------------------------------------------
  -- Output logic block
  -- Y = f(S) 
  ----------------------------------------------------------------
  output_logic : process (current_state)
  begin
    case current_state is
      ----------------------------------------------------------------
      when reset_1 =>
	--
	-- load zero in pc, tbr, iar, mdr, mar / init memory interface
	--
	s1_enab <= s1_const after tpd_fsm;       
        s2_enab <= s2_none after tpd_fsm;            
        alu_op_sel <= alu_pass_s1 after tpd_fsm;
        dest_enab <= dest_res after tpd_fsm;     
        const_sel <= const_00 after tpd_fsm;
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_dcare after tpd_fsm;    -- don't care
        exc_enab <= exc_none after tpd_fsm;     
        mem_ctrl <= mem_none after tpd_fsm;
      ----------------------------------------------------------------
      when reset_2 =>
	--
	-- load one into icr (S <- 1) / init memory interface
	--
	s1_enab <= s1_none after tpd_fsm;    
        s2_enab <= s2_none after tpd_fsm;            
        alu_op_sel <= alu_dcare after tpd_fsm;   -- don't care
        dest_enab <= dest_none after tpd_fsm;    
        const_sel <= const_dcare after tpd_fsm;  -- don't care
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_dcare after tpd_fsm;    -- don't care
        exc_enab <= exc_sbit after tpd_fsm;     
        mem_ctrl <= mem_none after tpd_fsm;
      ----------------------------------------------------------------
      when fetch =>
	--
	-- fetch instruction 
	--
	s1_enab <= s1_none after tpd_fsm;            
        s2_enab <= s2_none after tpd_fsm;                  
        alu_op_sel <= alu_dcare after tpd_fsm;   -- dont' care      
        dest_enab <= dest_none after tpd_fsm;
        const_sel <= const_dcare after tpd_fsm;  -- don't care
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_dcare after tpd_fsm;    -- don't care
        exc_enab <= exc_none after tpd_fsm;      
        mem_ctrl <= mem_fetch after tpd_fsm;    
      ----------------------------------------------------------------
      when decode_pcincr_ab =>
	--
	-- pc <- pc + 4 / load ab from reg.file / decode instruction
	--
	s1_enab <= s1_const after tpd_fsm;        
        s2_enab <= s2_pc after tpd_fsm;                 
        alu_op_sel <= alu_s1_add_s2 after tpd_fsm;       
        dest_enab <= dest_pc after tpd_fsm;
        const_sel <= const_04 after tpd_fsm;    
	rf_op_sel <= rfop_ab_rf after tpd_fsm;
	immed_sel <= imm_dcare after tpd_fsm;        -- don't care
        exc_enab <= exc_none after tpd_fsm;          
        mem_ctrl <= mem_none after tpd_fsm;    
      ----------------------------------------------------------------
      when memory =>
	--
	-- mar <- a + y (y = b , imm_s16)
	-- load (a)b from reg. file (in case instruction is Rtype store)
	--
	s1_enab <= s1_a after tpd_fsm;       
        s2_enab <= s2_Y after tpd_fsm;
        alu_op_sel <= alu_s1_add_s2 after tpd_fsm;     
        dest_enab <= dest_mar after tpd_fsm;
        const_sel <= const_dcare after tpd_fsm;       -- don't care
	rf_op_sel <= rfop_ab_rfx after tpd_fsm;
	immed_sel <= imm_s16 after tpd_fsm;       
        exc_enab <= exc_none after tpd_fsm;           
        mem_ctrl <= mem_ldst after tpd_fsm;    
      ----------------------------------------------------------------
      when load_w_1 =>
	--
	-- mdr <- mem
	--
	s1_enab <= s1_none after tpd_fsm;              
        s2_enab <= s2_none after tpd_fsm;        
        alu_op_sel <= alu_dcare after tpd_fsm;      -- don't care   
        dest_enab <= dest_mdr after tpd_fsm;
        const_sel <= const_dcare after tpd_fsm;     -- don't care
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_dcare after tpd_fsm;       -- don't care
        exc_enab <= exc_none after tpd_fsm;     
        mem_ctrl <= mem_lw after tpd_fsm;           
      ----------------------------------------------------------------
      when load_w_2 =>
	--
	-- write back for load word : c <- mdr
	--
	s1_enab <= s1_mdr after tpd_fsm;           
        s2_enab <= s2_none after tpd_fsm;                
        alu_op_sel <= alu_pass_s1 after tpd_fsm;     
        dest_enab <= dest_c after tpd_fsm;
        const_sel <= const_dcare after tpd_fsm;        -- don't care      
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_dcare after tpd_fsm;          -- don't care
        exc_enab <= exc_none after tpd_fsm;
        mem_ctrl <= mem_none after tpd_fsm;    
      ----------------------------------------------------------------
      when load_h_1 =>
	--
	-- mdr <- mem  
	--
	s1_enab <= s1_none after tpd_fsm;              
        s2_enab <= s2_none after tpd_fsm;       
        alu_op_sel <= alu_dcare after tpd_fsm;      -- don't care   
        dest_enab <= dest_mdr after tpd_fsm;
        const_sel <= const_dcare after tpd_fsm;     -- don't care
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_dcare after tpd_fsm;       -- don't care
        exc_enab <= exc_none after tpd_fsm;     
        mem_ctrl <= mem_lh after tpd_fsm;    
      ----------------------------------------------------------------
      when load_h_2 =>
	--
	-- write back for load half : c <- sra(mdr,16) 
	--
	s1_enab <= s1_mdr after tpd_fsm;           
        s2_enab <= s2_const after tpd_fsm;     
        alu_op_sel <= alu_sra_s1_s2 after tpd_fsm;     
        dest_enab <= dest_c after tpd_fsm;
        const_sel <= const_10 after tpd_fsm;      
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_dcare after tpd_fsm;         -- don't care
        exc_enab <= exc_none after tpd_fsm;   
        mem_ctrl <= mem_none after tpd_fsm;    
      ----------------------------------------------------------------
      when load_b_1 =>
	--
	-- mdr <- mem 
	--
	s1_enab <= s1_none after tpd_fsm;         
        s2_enab <= s2_none after tpd_fsm;  
        alu_op_sel <= alu_dcare after tpd_fsm;       -- don't care   
        dest_enab <= dest_mdr after tpd_fsm;
        const_sel <= const_dcare after tpd_fsm;      -- don't care
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_dcare after tpd_fsm;        -- don't care
        exc_enab <= exc_none after tpd_fsm;      
        mem_ctrl <= mem_lb after tpd_fsm;    
      ----------------------------------------------------------------
      when load_b_2 =>
	--
	-- write back for load byte : c <- sra(mdr,24)
	--
	s1_enab <= s1_mdr after tpd_fsm;           
        s2_enab <= s2_const after tpd_fsm;     
        alu_op_sel <= alu_sra_s1_s2 after tpd_fsm;     
        dest_enab <= dest_c after tpd_fsm;
        const_sel <= const_18 after tpd_fsm;      
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_dcare after tpd_fsm;         -- don't care
        exc_enab <= exc_none after tpd_fsm;     
        mem_ctrl <= mem_none after tpd_fsm;    
     ----------------------------------------------------------------
      when store_w_1 =>
	--
	-- mdr <- b : move b into MDR via s2 bus
	--
	s1_enab <= s1_none after tpd_fsm;        
        s2_enab <= s2_b after tpd_fsm;      
        alu_op_sel <= alu_pass_s2 after tpd_fsm;    
        dest_enab <= dest_mdr after tpd_fsm;
        const_sel <= const_dcare after tpd_fsm;     -- don't care
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_dcare after tpd_fsm;       -- don't care
        exc_enab <= exc_none after tpd_fsm;    
        mem_ctrl <= mem_ldst after tpd_fsm;    
     ----------------------------------------------------------------
      when store_w_2 =>
	--
	-- write word : mem <- mdr
	--
	s1_enab <= s1_none after tpd_fsm;         
        s2_enab <= s2_none after tpd_fsm;       
        alu_op_sel <= alu_dcare after tpd_fsm;      -- don't care
        dest_enab <= dest_none after tpd_fsm;
        const_sel <= const_dcare after tpd_fsm;     -- don't care
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_dcare after tpd_fsm;       -- don't care
        exc_enab <= exc_none after tpd_fsm;    
        mem_ctrl <= mem_sw after tpd_fsm;    
      ----------------------------------------------------------------
      when store_h_1 =>
	--
	-- mdr <- b : move b into MDR via s2 bus
	--
	s1_enab <= s1_none after tpd_fsm;             
        s2_enab <= s2_b after tpd_fsm;      
        alu_op_sel <= alu_pass_s2 after tpd_fsm;    
        dest_enab <= dest_mdr after tpd_fsm;
        const_sel <= const_dcare after tpd_fsm;     -- don't care
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_dcare after tpd_fsm;       -- don't care
        exc_enab <= exc_none after tpd_fsm;     
        mem_ctrl <= mem_ldst after tpd_fsm;    
      ----------------------------------------------------------------
      when store_h_2 =>
	--
	-- write half : mem <- mdr
	--
	s1_enab <= s1_none after tpd_fsm;             
        s2_enab <= s2_none after tpd_fsm;       
        alu_op_sel <= alu_dcare after tpd_fsm;     -- don't care   
        dest_enab <= dest_none after tpd_fsm;
        const_sel <= const_dcare after tpd_fsm;    -- don't care 
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_dcare after tpd_fsm;      -- don't care
        exc_enab <= exc_none after tpd_fsm;      
        mem_ctrl <= mem_sh after tpd_fsm;    
      ----------------------------------------------------------------
      when store_b_1 =>
	--
	-- mdr <- b : move b into MDR via s2 bus
	--
	s1_enab <= s1_none after tpd_fsm;              
        s2_enab <= s2_b after tpd_fsm;
        alu_op_sel <= alu_pass_s2 after tpd_fsm;    
        dest_enab <= dest_mdr after tpd_fsm;
        const_sel <= const_dcare after tpd_fsm;    -- don't care
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_dcare after tpd_fsm;      -- don't care
        exc_enab <= exc_none after tpd_fsm;    
        mem_ctrl <= mem_ldst after tpd_fsm;    
      ----------------------------------------------------------------
      when store_b_2 =>
	--
	-- write byte : mem <- mdr
	--
	s1_enab <= s1_none after tpd_fsm;               
        s2_enab <= s2_none after tpd_fsm;
        alu_op_sel <= alu_dcare after tpd_fsm;    -- don't care   
        dest_enab <= dest_none after tpd_fsm;
        const_sel <= const_dcare after tpd_fsm;   -- don't care 
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_dcare after tpd_fsm;     -- don't care
        exc_enab <= exc_none after tpd_fsm;
        mem_ctrl <= mem_sb after tpd_fsm;    
      ----------------------------------------------------------------
      when movi2s =>
	--
	-- move gpr to S : X <- a (X = iar, icr, tbr)
	--
	s1_enab <= s1_a after tpd_fsm;              
        s2_enab <= s2_none after tpd_fsm;        
        alu_op_sel <= alu_pass_s1 after tpd_fsm;     
        dest_enab <= dest_X after tpd_fsm;          
        const_sel <= const_dcare after tpd_fsm;     -- don't care 
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_dcare after tpd_fsm;       -- don't care
        exc_enab <= exc_none after tpd_fsm;    
        mem_ctrl <= mem_none after tpd_fsm;    
      ----------------------------------------------------------------
      when movs2i =>
	--
	-- move S to gpr : c <- X (X = iar, icr, tbr) 
	--
	s1_enab <= s1_X after tpd_fsm;         
        s2_enab <= s2_none after tpd_fsm;             
        alu_op_sel <= alu_pass_s1 after tpd_fsm;     
        dest_enab <= dest_c after tpd_fsm;          
        const_sel <= const_dcare after tpd_fsm;     -- don't care 
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_dcare after tpd_fsm;       -- don't care
        exc_enab <= exc_none after tpd_fsm;    
        mem_ctrl <= mem_none after tpd_fsm;    
      ----------------------------------------------------------------
      when br_eqz | br_nez =>
	--
	-- branch if equal zero : a - 0 = 0 ?
	--
	s1_enab <= s1_a after tpd_fsm;              
        s2_enab <= s2_const after tpd_fsm;          
        alu_op_sel <= alu_s1_sub_s2 after tpd_fsm;        
        dest_enab <= dest_none after tpd_fsm;
        const_sel <= const_00 after tpd_fsm;          
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_dcare after tpd_fsm;          -- don't care
        exc_enab <= exc_none after tpd_fsm;      
        mem_ctrl <= mem_none after tpd_fsm;    
      ----------------------------------------------------------------
      when branch =>
	--
	-- execute branch : temp(mar) <- pc + imm_s16
	--
	s1_enab <= s1_immed after tpd_fsm;              
        s2_enab <= s2_pc after tpd_fsm;          
        alu_op_sel <= alu_s1_add_s2 after tpd_fsm;        
        dest_enab <= dest_mar after tpd_fsm;
        const_sel <= const_dcare after tpd_fsm;         -- don't care 
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_s16 after tpd_fsm;          
        exc_enab <= exc_none after tpd_fsm;          
        mem_ctrl <= mem_none after tpd_fsm;    
      ----------------------------------------------------------------
      when jump =>
	--
	-- calculate jump addr: temp(mar) <- pc + imm_s26
	--
	s1_enab <= s1_immed after tpd_fsm;              
        s2_enab <= s2_pc after tpd_fsm;          
        alu_op_sel <= alu_s1_add_s2 after tpd_fsm;        
        dest_enab <= dest_mar after tpd_fsm;
        const_sel <= const_dcare after tpd_fsm;           -- don't care 
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_s26 after tpd_fsm;       
        exc_enab <= exc_none after tpd_fsm;     
        mem_ctrl <= mem_none after tpd_fsm;    
      ----------------------------------------------------------------
      when jump_reg =>
	--
	-- calculate jump addr: temp(mar) <- a 
	--
	s1_enab <= s1_a after tpd_fsm;              
        s2_enab <= s2_none after tpd_fsm;               
        alu_op_sel <= alu_pass_s1 after tpd_fsm;        
        dest_enab <= dest_mar after tpd_fsm;
        const_sel <= const_dcare after tpd_fsm;          -- don't care 
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_dcare after tpd_fsm;            -- don't care
        exc_enab <= exc_none after tpd_fsm;   
        mem_ctrl <= mem_none after tpd_fsm;    
      ----------------------------------------------------------------
      when jump_lnk_1 | jump_lnk_reg_1 =>
	--
	-- save pc : r31 <- pc
	--
	s1_enab <= s1_none after tpd_fsm;              
        s2_enab <= s2_pc after tpd_fsm;                    
        alu_op_sel <= alu_pass_s2 after tpd_fsm;        
        dest_enab <= dest_c after tpd_fsm;
        const_sel <= const_dcare after tpd_fsm;        -- don't care 
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_dcare after tpd_fsm;          -- don't care
        exc_enab <= exc_none after tpd_fsm;     
        mem_ctrl <= mem_none after tpd_fsm;    
       ----------------------------------------------------------------
      when jump_lnk_2 =>
	--
	-- execute jump : temp(mar) <- pc + imm_s26 / write back c
	--
	s1_enab <= s1_immed after tpd_fsm;              
        s2_enab <= s2_pc after tpd_fsm;          
        alu_op_sel <= alu_s1_add_s2 after tpd_fsm;        
        dest_enab <= dest_mar after tpd_fsm;
        const_sel <= const_dcare after tpd_fsm;         -- don't care 
	rf_op_sel <= rfop_rf_c after tpd_fsm;
	immed_sel <= imm_s26 after tpd_fsm;       
        exc_enab <= exc_none after tpd_fsm;    
        mem_ctrl <= mem_none after tpd_fsm;    
     ----------------------------------------------------------------
      when jump_lnk_reg_2 =>
	--
	-- calculate jump addr: temp(mar) <- a / write back c
	--
	s1_enab <= s1_a after tpd_fsm;              
        s2_enab <= s2_none after tpd_fsm;           
        alu_op_sel <= alu_pass_s1 after tpd_fsm;        
        dest_enab <= dest_mar after tpd_fsm;
        const_sel <= const_dcare after tpd_fsm;         -- don't care 
	rf_op_sel <= rfop_rf_c after tpd_fsm;
	immed_sel <= imm_dcare after tpd_fsm;           -- don't care   
        exc_enab <= exc_none after tpd_fsm;   
        mem_ctrl <= mem_none after tpd_fsm;    
      ----------------------------------------------------------------
      when trap_1 =>
	--
	-- software interrupt : iar <- pc
	--
	s1_enab <= s1_none after tpd_fsm;          
        s2_enab <= s2_pc after tpd_fsm;          
        alu_op_sel <= alu_pass_s2 after tpd_fsm;       
        dest_enab <= dest_iar after tpd_fsm;    
        const_sel <= const_dcare after tpd_fsm;      -- don't care
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_dcare after tpd_fsm;        -- don't care
        exc_enab <= exc_none after tpd_fsm;     
        mem_ctrl <= mem_none after tpd_fsm;   
      ----------------------------------------------------------------
      when trap_2 =>
	--
	--  software interrupt : temp(mar) <- tbr + imm_s26 
	--
	s1_enab <= s1_tbr after tpd_fsm;          
        s2_enab <= s2_immed after tpd_fsm;      
        alu_op_sel <= alu_s1_add_s2 after tpd_fsm;       
        dest_enab <= dest_mar after tpd_fsm;       
        const_sel <= const_dcare after tpd_fsm;           -- don't care  
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_s26 after tpd_fsm;        
        exc_enab <= exc_none after tpd_fsm;     
        mem_ctrl <= mem_none after tpd_fsm;    
      ----------------------------------------------------------------
      when trap_3 =>
	--
	--  software intrpt., shift left logical icr : icr <- sll(icr,16)
	--  set S bit in icr (prior to load from dest_bus)
	--
	s1_enab <= s1_icr after tpd_fsm;
        s2_enab <= s2_const after tpd_fsm;      
        alu_op_sel <= alu_sll_s1_s2 after tpd_fsm;       
        dest_enab <= dest_icr after tpd_fsm;    
        const_sel <= const_10 after tpd_fsm;        
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_dcare after tpd_fsm;         -- don't care
        exc_enab <= exc_sbit after tpd_fsm;     
        mem_ctrl <= mem_none after tpd_fsm;    
      ----------------------------------------------------------------
      when rfe_1 =>
	--
	-- return from interrupt : shift right logical icr
	--
	s1_enab <= s1_icr after tpd_fsm;
        s2_enab <= s2_const after tpd_fsm;      
        alu_op_sel <= alu_srl_s1_s2 after tpd_fsm;       
        dest_enab <= dest_icr after tpd_fsm;        
        const_sel <= const_10 after tpd_fsm;        
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_dcare after tpd_fsm;         -- don't care
        exc_enab <= exc_none after tpd_fsm;             
        mem_ctrl <= mem_none after tpd_fsm;    
      ----------------------------------------------------------------
      when rfe_2 =>
	--
	-- return from interrupt : pc <- iar
	--
	s1_enab <= s1_iar after tpd_fsm;      
        s2_enab <= s2_none after tpd_fsm;             
        alu_op_sel <= alu_pass_s1 after tpd_fsm;       
        dest_enab <= dest_pc after tpd_fsm;     
        const_sel <= const_dcare after tpd_fsm;       -- don't care
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_dcare after tpd_fsm;         -- don't care
        exc_enab <= exc_none after tpd_fsm;     
        mem_ctrl <= mem_none after tpd_fsm;   
      ----------------------------------------------------------------
      when load_pc =>
	--
	-- load pc from temp. reg (mar): pc <- mar 
	-- (avoids illegal value in pc)
	--
	s1_enab <= s1_none after tpd_fsm;                   
        s2_enab <= s2_mar after tpd_fsm;      
        alu_op_sel <= alu_pass_s2 after tpd_fsm;        
        dest_enab <= dest_pc after tpd_fsm;
        const_sel <= const_dcare after tpd_fsm;       -- don't care 
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_dcare after tpd_fsm;         -- don't care
        exc_enab <= exc_none after tpd_fsm;     
        mem_ctrl <= mem_none after tpd_fsm;    
      ----------------------------------------------------------------
      when lhi =>
	--
	-- lhi: c <- sll(immed,16) 
	--
	s1_enab <= s1_immed after tpd_fsm;     
        s2_enab <= s2_const after tpd_fsm;     
        alu_op_sel <= alu_sll_s1_s2 after tpd_fsm;       
        dest_enab <= dest_c after tpd_fsm;      
        const_sel <= const_10 after tpd_fsm;        
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_dcare after tpd_fsm;       -- don't care      
        exc_enab <= exc_none after tpd_fsm;          
        mem_ctrl <= mem_none after tpd_fsm;   
      ----------------------------------------------------------------
      when add_1 =>
	--
	-- add: c <- a + y / y : b or immed 
	--
	s1_enab <= s1_a after tpd_fsm;     
        s2_enab <= s2_Y after tpd_fsm;     
        alu_op_sel <= alu_s1_add_s2 after tpd_fsm;       
        dest_enab <= dest_c after tpd_fsm;          
        const_sel <= const_dcare after tpd_fsm;        -- don't care
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_s16 after tpd_fsm;         
        exc_enab <= exc_none after tpd_fsm;              
        mem_ctrl <= mem_none after tpd_fsm;   
      ----------------------------------------------------------------
      when sub_1 =>
	--
	-- sub: c <- a - y / y : b or immed 
	--
	s1_enab <= s1_a after tpd_fsm;     
        s2_enab <= s2_Y after tpd_fsm;     
        alu_op_sel <= alu_s1_sub_s2 after tpd_fsm;       
        dest_enab <= dest_c after tpd_fsm;          
        const_sel <= const_dcare after tpd_fsm;        -- don't care
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_s16 after tpd_fsm;         
        exc_enab <= exc_none after tpd_fsm;               
        mem_ctrl <= mem_none after tpd_fsm;   
     ----------------------------------------------------------------
      when and_1 =>
	--
	-- and: c <- a and y / y : b or immed 
	--
	s1_enab <= s1_a after tpd_fsm;     
        s2_enab <= s2_Y after tpd_fsm;     
        alu_op_sel <= alu_s1_and_s2 after tpd_fsm;       
        dest_enab <= dest_c after tpd_fsm;          
        const_sel <= const_dcare after tpd_fsm;        -- don't care
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_u16 after tpd_fsm;         
        exc_enab <= exc_none after tpd_fsm;          
        mem_ctrl <= mem_none after tpd_fsm;   
     ----------------------------------------------------------------
      when or_1 =>
	--
	-- or: c <- a or y / y : b or immed  
	--
	s1_enab <= s1_a after tpd_fsm;     
        s2_enab <= s2_Y after tpd_fsm;     
        alu_op_sel <= alu_s1_or_s2 after tpd_fsm;       
        dest_enab <= dest_c after tpd_fsm;          
        const_sel <= const_dcare after tpd_fsm;        -- don't care
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_u16 after tpd_fsm;         
        exc_enab <= exc_none after tpd_fsm;          
        mem_ctrl <= mem_none after tpd_fsm;   
      ----------------------------------------------------------------
      when xor_1 =>
	--
	-- xor: c <- a xor y / y : b or immed 
	--
	s1_enab <= s1_a after tpd_fsm;     
        s2_enab <= s2_Y after tpd_fsm;     
        alu_op_sel <= alu_s1_xor_s2 after tpd_fsm;       
        dest_enab <= dest_c after tpd_fsm;          
        const_sel <= const_dcare after tpd_fsm;        -- don't care
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_u16 after tpd_fsm;         
        exc_enab <= exc_none after tpd_fsm;          
        mem_ctrl <= mem_none after tpd_fsm;   
      ----------------------------------------------------------------
      when sll_1 =>
	--
	-- sll: c <- a sll y / y : b or immed
	--
	s1_enab <= s1_a after tpd_fsm;     
        s2_enab <= s2_Y after tpd_fsm;     
        alu_op_sel <= alu_sll_s1_s2 after tpd_fsm;       
        dest_enab <= dest_c after tpd_fsm;          
        const_sel <= const_dcare after tpd_fsm;        -- don't care
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_s16 after tpd_fsm;         
        exc_enab <= exc_none after tpd_fsm;          
        mem_ctrl <= mem_none after tpd_fsm;   
     ----------------------------------------------------------------
      when srl_1 =>
 	--
	-- srl: c <- a srl y / y : b or immed
	--
	s1_enab <= s1_a after tpd_fsm;     
        s2_enab <= s2_Y after tpd_fsm;     
        alu_op_sel <= alu_srl_s1_s2 after tpd_fsm;       
        dest_enab <= dest_c after tpd_fsm;          
        const_sel <= const_dcare after tpd_fsm;        -- don't care
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_s16 after tpd_fsm;      
        exc_enab <= exc_none after tpd_fsm;          
        mem_ctrl <= mem_none after tpd_fsm;   
     ----------------------------------------------------------------
      when sra_1 =>
	--
	-- sra: c <- a sra y / y : b or immed
	--
	s1_enab <= s1_a after tpd_fsm;     
        s2_enab <= s2_Y after tpd_fsm;     
        alu_op_sel <= alu_sra_s1_s2 after tpd_fsm;       
        dest_enab <= dest_c after tpd_fsm;          
        const_sel <= const_dcare after tpd_fsm;        -- don't care
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_s16 after tpd_fsm;         
        exc_enab <= exc_none after tpd_fsm;          
        mem_ctrl <= mem_none after tpd_fsm;   
      ----------------------------------------------------------------
      when seq_1 | sne_1 | slt_1 | sle_1 | sgt_1 | sge_1 =>
	--
	-- set if s1 equal s2 
 	-- none <- a sub y / y : b or immed 
	--
	s1_enab <= s1_a after tpd_fsm;     
        s2_enab <= s2_Y after tpd_fsm;     
        alu_op_sel <= alu_s1_sub_s2 after tpd_fsm;       
        dest_enab <= dest_none after tpd_fsm;
        const_sel <= const_dcare after tpd_fsm;        -- don't care
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_s16 after tpd_fsm;        
        exc_enab <= exc_none after tpd_fsm;          
        mem_ctrl <= mem_none after tpd_fsm;   
      ----------------------------------------------------------------
      when set_to_1 =>
  	--
	-- c <- X"0000_0001"
	--
	s1_enab <= s1_const after tpd_fsm;        
        s2_enab <= s2_none after tpd_fsm;     
        alu_op_sel <= alu_pass_s1 after tpd_fsm;       
        dest_enab <= dest_c after tpd_fsm;     
        const_sel <= const_01 after tpd_fsm;       
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_dcare after tpd_fsm;      -- don't care      
        exc_enab <= exc_none after tpd_fsm;          
        mem_ctrl <= mem_none after tpd_fsm;   
      ----------------------------------------------------------------
      when set_to_0 =>
	--
	-- c <- X"0000_0000"
	--
	s1_enab <= s1_const after tpd_fsm;
        s2_enab <= s2_none after tpd_fsm;     
        alu_op_sel <= alu_pass_s1 after tpd_fsm;       
        dest_enab <= dest_c after tpd_fsm;     
        const_sel <= const_00 after tpd_fsm;       
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_dcare after tpd_fsm;      -- don't care      
        exc_enab <= exc_none after tpd_fsm;          
        mem_ctrl <= mem_none after tpd_fsm;   
      ----------------------------------------------------------------
      when write_back =>
	--
	-- write back (reg.reg.alu-instr): rf <- c 
	--
	s1_enab <= s1_none after tpd_fsm;                 
        s2_enab <= s2_none after tpd_fsm;              
        alu_op_sel <= alu_dcare after tpd_fsm;   -- don't care   
        dest_enab <= dest_none after tpd_fsm;
        const_sel <= const_dcare after tpd_fsm;       -- don't care      
	rf_op_sel <= rfop_rf_c after tpd_fsm;
	immed_sel <= imm_dcare after tpd_fsm;        -- don't care
        exc_enab <= exc_none after tpd_fsm;    
        mem_ctrl <= mem_none after tpd_fsm;    
      ----------------------------------------------------------------
      when intrpt_1 =>
 	--
	--  interrupt : shift left logical icr
	--
	s1_enab <= s1_icr after tpd_fsm;
        s2_enab <= s2_const after tpd_fsm;      
        alu_op_sel <= alu_sll_s1_s2 after tpd_fsm;       
        dest_enab <= dest_icr after tpd_fsm;        
        const_sel <= const_10 after tpd_fsm;        
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_dcare after tpd_fsm;         -- don't care
        exc_enab <= exc_none after tpd_fsm;     
        mem_ctrl <= mem_none after tpd_fsm;    
      ----------------------------------------------------------------
      when intrpt_2 =>
	--
	-- interrupt : iar <- pc
	--
	s1_enab <= s1_none after tpd_fsm;     
        s2_enab <= s2_pc after tpd_fsm;        
        alu_op_sel <= alu_pass_s2 after tpd_fsm;       
        dest_enab <= dest_iar after tpd_fsm;        
        const_sel <= const_dcare after tpd_fsm;       -- don't care
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_dcare after tpd_fsm;         -- don't care
        exc_enab <= exc_none after tpd_fsm;     
        mem_ctrl <= mem_none after tpd_fsm;
      ----------------------------------------------------------------
      when intrpt_3 =>
	--
	--  interrupt : pc <- tbr
	--  set S-bit
	--
	s1_enab <= s1_tbr after tpd_fsm;     
        s2_enab <= s2_none after tpd_fsm;               
        alu_op_sel <= alu_pass_s1 after tpd_fsm;       
        dest_enab <= dest_pc after tpd_fsm;         
        const_sel <= const_dcare after tpd_fsm;       -- don't care        
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_dcare after tpd_fsm;         -- don't care
        exc_enab <= exc_sbit after tpd_fsm;     
        mem_ctrl <= mem_none after tpd_fsm;    
      ----------------------------------------------------------------
      when ill_opc =>
	--
 	--  illegal opcode: shift left logical icr : icr <- sll(icr,16)
	--  set IOC bit in icr (prior to load from dest_bus)
	--
	s1_enab <= s1_icr after tpd_fsm;
        s2_enab <= s2_const after tpd_fsm;      
        alu_op_sel <= alu_sll_s1_s2 after tpd_fsm;       
        dest_enab <= dest_icr after tpd_fsm;    
        const_sel <= const_10 after tpd_fsm;        
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_dcare after tpd_fsm;         -- don't care
        exc_enab <= exc_ioc after tpd_fsm;     
        mem_ctrl <= mem_none after tpd_fsm;    
      ----------------------------------------------------------------
      when ill_rraf =>
	--
 	--  illegal reg.reg.alu function: shift left logical icr : icr <- sll(icr,16)
	--  set IRRA bit in icr (prior to load from dest_bus)
	--
	s1_enab <= s1_icr after tpd_fsm;
        s2_enab <= s2_const after tpd_fsm;      
        alu_op_sel <= alu_sll_s1_s2 after tpd_fsm;       
        dest_enab <= dest_icr after tpd_fsm;    
        const_sel <= const_10 after tpd_fsm;        
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_dcare after tpd_fsm;         -- don't care
        exc_enab <= exc_irra after tpd_fsm;     
        mem_ctrl <= mem_none after tpd_fsm;    
      ----------------------------------------------------------------
      when d_adr_viol =>
	--
 	--  data address violation: shift left logical icr : icr <- sll(icr,16)
	--  set DAV bit in icr (prior to load from dest_bus)
	-- 
	s1_enab <= s1_icr after tpd_fsm;
        s2_enab <= s2_const after tpd_fsm;      
        alu_op_sel <= alu_sll_s1_s2 after tpd_fsm;       
        dest_enab <= dest_icr after tpd_fsm;    
        const_sel <= const_10 after tpd_fsm;        
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_dcare after tpd_fsm;         -- don't care
        exc_enab <= exc_dav after tpd_fsm;     
        mem_ctrl <= mem_none after tpd_fsm;    
      ----------------------------------------------------------------
      when i_adr_viol =>
	--
 	--  instr. address violation: shift left logical icr : icr <- sll(icr,16)
	--  set IAV bit in icr (prior to load from dest_bus)
	-- 
	s1_enab <= s1_icr after tpd_fsm;
        s2_enab <= s2_const after tpd_fsm;      
        alu_op_sel <= alu_sll_s1_s2 after tpd_fsm;       
        dest_enab <= dest_icr after tpd_fsm;    
        const_sel <= const_10 after tpd_fsm;        
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_dcare after tpd_fsm;         -- don't care
        exc_enab <= exc_iav after tpd_fsm;     
        mem_ctrl <= mem_none after tpd_fsm;    
      ----------------------------------------------------------------
      when priv_viol =>
	-- 
 	--  priviledge violation: shift left logical icr : icr <- sll(icr,16)
	--  set PRIV bit in icr (prior to load from dest_bus)
	-- 
	s1_enab <= s1_icr after tpd_fsm;
        s2_enab <= s2_const after tpd_fsm;      
        alu_op_sel <= alu_sll_s1_s2 after tpd_fsm;       
        dest_enab <= dest_icr after tpd_fsm;    
        const_sel <= const_10 after tpd_fsm;        
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_dcare after tpd_fsm;         -- don't care
        exc_enab <= exc_priv after tpd_fsm;     
        mem_ctrl <= mem_none after tpd_fsm;    
      ----------------------------------------------------------------
      when except_ovad =>
	-- 
 	--  address overflow: shift left logical icr : icr <- sll(icr,16)
	--  set OVAD bit in icr (prior to load from dest_bus)
	-- 
	s1_enab <= s1_icr after tpd_fsm;
        s2_enab <= s2_const after tpd_fsm;      
        alu_op_sel <= alu_sll_s1_s2 after tpd_fsm;       
        dest_enab <= dest_icr after tpd_fsm;    
        const_sel <= const_10 after tpd_fsm;        
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_dcare after tpd_fsm;         -- don't care
        exc_enab <= exc_ovad after tpd_fsm;     
        mem_ctrl <= mem_none after tpd_fsm;    
      ----------------------------------------------------------------
      when except_ovar =>
	-- 
 	--  arithmetic overflow: shift left logical icr : icr <- sll(icr,16)
	--  set OVAR bit in icr (prior to load from dest_bus)
	-- 
	s1_enab <= s1_icr after tpd_fsm;
        s2_enab <= s2_const after tpd_fsm;      
        alu_op_sel <= alu_sll_s1_s2 after tpd_fsm;       
        dest_enab <= dest_icr after tpd_fsm;    
        const_sel <= const_10 after tpd_fsm;        
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_dcare after tpd_fsm;         -- don't care
        exc_enab <= exc_ovar after tpd_fsm;     
        mem_ctrl <= mem_none after tpd_fsm;    
      ----------------------------------------------------------------
      when halt_state =>
	--
	-- set all outputs to high impedance/ inouts to input 
	--
	s1_enab <= s1_none after tpd_fsm;        
        s2_enab <= s2_none after tpd_fsm;        
        alu_op_sel <= alu_dcare after tpd_fsm;      -- dont' care
        dest_enab <= dest_none after tpd_fsm;
        const_sel <= const_dcare after tpd_fsm;     -- don't care
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_dcare after tpd_fsm;       -- don't care
        exc_enab <= exc_none after tpd_fsm;     
        mem_ctrl <= mem_halt after tpd_fsm;    
      ----------------------------------------------------------------
      when error_state =>
	--
	-- set all outputs to high impedance/ inouts to input / set error  
	--
	s1_enab <= s1_none after tpd_fsm;  
        s2_enab <= s2_none after tpd_fsm;         
        alu_op_sel <= alu_dcare after tpd_fsm;      -- dont' care
        dest_enab <= dest_none after tpd_fsm;
        const_sel <= const_dcare after tpd_fsm;     -- don't care
	rf_op_sel <= rfop_none after tpd_fsm;
	immed_sel <= imm_dcare after tpd_fsm;       -- don't care
        exc_enab <= exc_none after tpd_fsm;     
        mem_ctrl <= mem_error after tpd_fsm;    
      ----------------------------------------------------------------
     end case;
  end process output_logic;
      
  --synopsys synthesis_off
  --------------------------------------------------------------------------
  -- Debugging procedure
  -- Print actual state to stdout
  --------------------------------------------------------------------------
  write_state : process(current_state)
      variable L : Line;
  begin
    if DEBUG then
      case current_state is
	when reset_1 => write(L, string'("-> fsm_state : reset_1"));
	when reset_2 => write(L, string'("-> fsm_state : reset_2"));	  
	when fetch => write(L, string'("-> fsm_state : fetch"));
	when decode_pcincr_ab => write(L, string'("-> fsm_state : decode_pcincr_ab"));
	when memory => write(L, string'("-> fsm_state : memory"));
	when load_w_1 => write(L, string'("-> fsm_state : load_w_1"));
	when load_w_2 => write(L, string'("-> fsm_state : load_w_2"));
	when load_h_1 => write(L, string'("-> fsm_state : load_h_1"));
	when load_h_2 => write(L, string'("-> fsm_state : load_h_2"));
	when load_b_1 => write(L, string'("-> fsm_state : load_b_1"));
	when load_b_2 => write(L, string'("-> fsm_state : load_b_2"));
	when store_w_1 => write(L, string'("-> fsm_state : store_w_1"));
	when store_w_2 => write(L, string'("-> fsm_state : store_w_2"));
	when store_h_1 => write(L, string'("-> fsm_state : store_h_1"));
	when store_h_2 => write(L, string'("-> fsm_state : store_h_2"));
	when store_b_1 => write(L, string'("-> fsm_state : store_b_1"));
	when store_b_2 => write(L, string'("-> fsm_state : store_b_2"));
	when lhi => write(L, string'("-> fsm_state : lhi"));
	when movs2i => write(L, string'("-> fsm_state : movs2i"));
 	when movi2s => write(L, string'("-> fsm_state : movi2s"));
 	when br_eqz => write(L, string'("-> fsm_state : br_eqz"));
	when br_nez => write(L, string'("-> fsm_state : br_nez"));
	when branch => write(L, string'("-> fsm_state : branch"));
	when jump => write(L, string'("-> fsm_state : jump"));
	when jump_lnk_1 => write(L, string'("-> fsm_state : jump_lnk_1"));
	when jump_lnk_2 => write(L, string'("-> fsm_state : jump_lnk_2"));
	when jump_reg => write(L, string'("-> fsm_state : jump_reg"));
	when jump_lnk_reg_1 => write(L, string'("-> fsm_state : jump_lnk_reg_1"));
	when jump_lnk_reg_2 => write(L, string'("-> fsm_state : jump_lnk_reg_2"));
	when trap_1 => write(L, string'("-> fsm_state : trap_1"));
	when trap_2 => write(L, string'("-> fsm_state : trap_2"));
	when trap_3 => write(L, string'("-> fsm_state : trap_3"));
	when rfe_1 => write(L, string'("-> fsm_state : rfe_1"));
	when rfe_2 => write(L, string'("-> fsm_state : rfe_2"));
	when load_pc => write(L, string'("-> fsm_state : load_pc"));	  
	when add_1 => write(L, string'("-> fsm_state : add_1"));
	when sub_1 => write(L, string'("-> fsm_state : sub_1"));
	when and_1 => write(L, string'("-> fsm_state : and_1")); 
	when or_1 => write(L, string'("-> fsm_state : or_1")); 
	when xor_1 => write(L, string'("-> fsm_state : xor_1")); 
	when sll_1 => write(L, string'("-> fsm_state : sll_1")); 
	when srl_1 => write(L, string'("-> fsm_state : srl_1")); 
	when sra_1 => write(L, string'("-> fsm_state : sra_1")); 
	when seq_1 => write(L, string'("-> fsm_state : seq_1")); 
	when sne_1 => write(L, string'("-> fsm_state : sne_1")); 
	when slt_1 => write(L, string'("-> fsm_state : slt_1")); 
	when sle_1 => write(L, string'("-> fsm_state : sle_1")); 
	when sgt_1 => write(L, string'("-> fsm_state : sgt_1")); 
	when sge_1 => write(L, string'("-> fsm_state : sge_1")); 
	when set_to_1 => write(L, string'("-> fsm_state : set_to_1")); 
	when set_to_0 => write(L, string'("-> fsm_state : set_to_0"));
	when write_back => write(L, string'("-> fsm_state : write_back")); 
	when intrpt_1 => write(L, string'("-> fsm_state : intrpt_1")); 
	when intrpt_2 => write(L, string'("-> fsm_state : intrpt_2")); 
	when intrpt_3 => write(L, string'("-> fsm_state : intrpt_3")); 
	when ill_opc => write(L, string'("-> fsm_state : ill_opc")); 
	when ill_rraf => write(L, string'("-> fsm_state : ill_rraf")); 
	when d_adr_viol => write(L, string'("-> fsm_state : d_adr_viol")); 
	when i_adr_viol => write(L, string'("-> fsm_state : i_adr_viol")); 
	when priv_viol => write(L, string'("-> fsm_state : priv_viol")); 
	when except_ovad => write(L, string'("-> fsm_state : except_ovad"));
	when except_ovar => write(L, string'("-> fsm_state : except_ovar"));
	when halt_state => write(L, string'("-> fsm_state : halt_state")); 
	when error_state => write(L, string'("-> fsm_state : error_state")); 
      end case;
      writeline(output, L);
    end if;
  end process write_state;
  --synopsys synthesis_on
  
end moore_2;
