--------------------------------------------------------------------------
--  DLX PROCESSOR MODEL SUITE
--  Copyright (C) 1995, Martin Gumm
--  University of Stuttgart / Department of Computer Science / IPVR-ISE
--------------------------------------------------------------------------
--  This program is free software; you can redistribute it and/or modify
--  it under the terms of the GNU General Public License as published by
--  the Free Software Foundation; either version 1, or (at your option)
--  any later version.
--
--  This program is distributed in the hope that it will be useful,
--  but WITHOUT ANY WARRANTY; without even the implied warranty of
--  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--  GNU General Public License for more details.
--------------------------------------------------------------------------
--  Last revision date : November 15 1995
--------------------------------------------------------------------------

--------------------------------------------------------------------------
--  *** SYNOPSYS synthesizable code (ver. 3.2.a) ***
--------------------------------------------------------------------------

--------------------------------------------------------------------------
--   Package specification for DLX instructions
--  
--  (file dlx-instructions.vhd)
--------------------------------------------------------------------------

--------------------------------------------------------------------------
-- A dlx instruction is 32 bits wide. There are three instruction formats:
--
-- I-type:
--  0         5 6      10 11     15 16                            31
-- +----------------------------------------------------------------+
-- |  opcode   |   rs1   |   rd    |           immed16              |
-- +----------------------------------------------------------------+
--
-- Encodes: - loads and stores of bytes, words, half-words
--            with immediate displacement
--          - all immediates (rd <- rs1 op immediate)
--          - conditional branch instructions (rd unused)
--          - jump register, jump and link register
--            (rd = 0; rs = dezstination; immediate = 0)
--  
-- R-type:
--  0         5 6      10 11     15 16     20 21      25 26       31
-- +----------------------------------------------------------------+
-- |  opcode   |   rs1   |   rs2   |   rd    |        func          |
-- +----------------------------------------------------------------+
--                                           | fp_func  | rr_func   |
--                                           +----------------------+
--
-- Encodes: - resister-register ALU operations (rd <- rs1 func rs2)
--            function encodes the datapath operation (add, sub, ...)
--            read/write special registers and moves 
--          - loads and stores of bytes, words, half-words
--            with register displacement
--
-- J-type:
--  0         5 6                                                 31
-- +----------------------------------------------------------------+
-- |  opcode   |                     immed26                        |
-- +----------------------------------------------------------------+
--
-- Encodes: - jump and jump-and-link
--          - trap and rfe
--            
--------------------------------------------------------------------------

library IEEE;
use IEEE.std_logic_1164.all;

use WORK.dlx_types.ALL;

--synopsys synthesis_off
use STD.textio.all;
--synopsys synthesis_on


package dlx_instructions is

  --
  -- encoding of opcodes
  --
  constant op_rr_alu	: dlx_opcode := "000000";
  constant op_lhu_i	: dlx_opcode := "000001";
  constant op_lbu_i	: dlx_opcode := "000010";
  constant op_ud_03	: dlx_opcode := "000011";
  constant op_lw_i	: dlx_opcode := "000100";
  constant op_lh_i	: dlx_opcode := "000101";
  constant op_lb_i	: dlx_opcode := "000110";
  constant op_ud_07	: dlx_opcode := "000111";
  constant op_sw_i	: dlx_opcode := "001000";
  constant op_sh_i	: dlx_opcode := "001001";
  constant op_sb_i	: dlx_opcode := "001010";
  constant op_ud_0B	: dlx_opcode := "001011";
  constant op_j 	: dlx_opcode := "001100";
  constant op_jal	: dlx_opcode := "001101";
  constant op_jr 	: dlx_opcode := "001110";
  constant op_jalr	: dlx_opcode := "001111";

  constant op_beqz	: dlx_opcode := "010000";
  constant op_bnez	: dlx_opcode := "010001";
  constant op_rfe	: dlx_opcode := "010010";
  constant op_trap	: dlx_opcode := "010011";
  constant op_add_i	: dlx_opcode := "010100";
  constant op_addu_i	: dlx_opcode := "010101";
  constant op_sub_i	: dlx_opcode := "010110";
  constant op_subu_i	: dlx_opcode := "010111";
  constant op_and_i	: dlx_opcode := "011000";
  constant op_or_i	: dlx_opcode := "011001";
  constant op_xor_i	: dlx_opcode := "011010";
  constant op_lhi	: dlx_opcode := "011011";
  constant op_sll_i	: dlx_opcode := "011100";
  constant op_ud_1D	: dlx_opcode := "011101";
  constant op_srl_i	: dlx_opcode := "011110";
  constant op_sra_i	: dlx_opcode := "011111";

  constant op_seq_i	: dlx_opcode := "100000";
  constant op_sequ_i	: dlx_opcode := "100001";
  constant op_sne_i	: dlx_opcode := "100010";
  constant op_sneu_i	: dlx_opcode := "100011";
  constant op_slt_i	: dlx_opcode := "100100";
  constant op_sltu_i	: dlx_opcode := "100101";
  constant op_sle_i	: dlx_opcode := "100110";
  constant op_sleu_i	: dlx_opcode := "100111";
  constant op_sgt_i	: dlx_opcode := "101000";
  constant op_sgtu_i	: dlx_opcode := "101001";
  constant op_sge_i	: dlx_opcode := "101010";
  constant op_sgeu_i	: dlx_opcode := "101011";
  constant op_ud_2C	: dlx_opcode := "101100";
  constant op_ud_2D	: dlx_opcode := "101101";
  constant op_ud_2E	: dlx_opcode := "101110";
  constant op_ud_2F	: dlx_opcode := "101111";

  constant op_lf_i	: dlx_opcode := "110000";
  constant op_ld_i	: dlx_opcode := "110001";
  constant op_ud_32	: dlx_opcode := "110010";
  constant op_ud_33	: dlx_opcode := "110011";
  constant op_sf_i	: dlx_opcode := "110100";
  constant op_sd_i	: dlx_opcode := "110101";
  constant op_ud_36	: dlx_opcode := "110110";
  constant op_ud_37	: dlx_opcode := "110111";
  constant op_bfpt	: dlx_opcode := "111000";
  constant op_bfpf	: dlx_opcode := "111001";
  constant op_ud_3A	: dlx_opcode := "111010";
  constant op_ud_3B	: dlx_opcode := "111011";
  constant op_fp_fn	: dlx_opcode := "111100";
  constant op_ud_3D	: dlx_opcode := "111101";
  constant op_ud_3E	: dlx_opcode := "111110";
  constant op_ud_3F	: dlx_opcode := "111111";

  --
  -- encoding of register-register ALU functions
  --
  constant rr_func_nop		: dlx_rr_func := "000000";
  constant rr_func_ud_01	: dlx_rr_func := "000001";
  constant rr_func_movi2s 	: dlx_rr_func := "000010";
  constant rr_func_movs2i 	: dlx_rr_func := "000011";
  constant rr_func_add	 	: dlx_rr_func := "000100";
  constant rr_func_addu 	: dlx_rr_func := "000101";
  constant rr_func_sub	 	: dlx_rr_func := "000110";
  constant rr_func_subu	 	: dlx_rr_func := "000111";
  constant rr_func_and	        : dlx_rr_func := "001000";
  constant rr_func_or	        : dlx_rr_func := "001001";
  constant rr_func_xor  	: dlx_rr_func := "001010";
  constant rr_func_ud_0B	: dlx_rr_func := "001011";
  constant rr_func_sll	        : dlx_rr_func := "001100";
  constant rr_func_ud_0D	: dlx_rr_func := "001101";
  constant rr_func_srl  	: dlx_rr_func := "001110";
  constant rr_func_sra  	: dlx_rr_func := "001111";

  constant rr_func_seq		: dlx_rr_func := "010000";
  constant rr_func_sequ		: dlx_rr_func := "010001";
  constant rr_func_sne		: dlx_rr_func := "010010";
  constant rr_func_sneu		: dlx_rr_func := "010011";
  constant rr_func_slt		: dlx_rr_func := "010100";
  constant rr_func_sltu		: dlx_rr_func := "010101";
  constant rr_func_sle  	: dlx_rr_func := "010110";
  constant rr_func_sleu 	: dlx_rr_func := "010111";
  constant rr_func_sgt  	: dlx_rr_func := "011000";
  constant rr_func_sgtu 	: dlx_rr_func := "011001";
  constant rr_func_sge  	: dlx_rr_func := "011010";
  constant rr_func_sgeu 	: dlx_rr_func := "011011";
  constant rr_func_movf	        : dlx_rr_func := "011100";
  constant rr_func_movd	        : dlx_rr_func := "011101";
  constant rr_func_movfp2i	: dlx_rr_func := "011110";
  constant rr_func_movi2fp	: dlx_rr_func := "011111";

  constant rr_func_lw           : dlx_rr_func := "100000";
  constant rr_func_lh           : dlx_rr_func := "100001";
  constant rr_func_lb           : dlx_rr_func := "100010";
  constant rr_func_ud_23        : dlx_rr_func := "100011";
  constant rr_func_ud_24        : dlx_rr_func := "100100";
  constant rr_func_lhu          : dlx_rr_func := "100101";
  constant rr_func_lbu          : dlx_rr_func := "100110";
  constant rr_func_ud_27        : dlx_rr_func := "100111";
  constant rr_func_sw           : dlx_rr_func := "101000";
  constant rr_func_sh           : dlx_rr_func := "101001";
  constant rr_func_sb           : dlx_rr_func := "101010";
  constant rr_func_ud_2B        : dlx_rr_func := "101011";
  constant rr_func_lf           : dlx_rr_func := "101100";
  constant rr_func_ld           : dlx_rr_func := "101101";
  constant rr_func_ud_2E        : dlx_rr_func := "101110";
  constant rr_func_ud_2F        : dlx_rr_func := "101111";
  constant rr_func_sf           : dlx_rr_func := "110000";
  constant rr_func_sd           : dlx_rr_func := "110001";
  constant rr_func_ud_32        : dlx_rr_func := "110010";
  constant rr_func_ud_33        : dlx_rr_func := "110011";
  constant rr_func_ud_34        : dlx_rr_func := "110100";
  constant rr_func_ud_35        : dlx_rr_func := "110101";
  constant rr_func_ud_36        : dlx_rr_func := "110110";
  constant rr_func_ud_37        : dlx_rr_func := "110111";
  constant rr_func_ud_38        : dlx_rr_func := "111000";
  constant rr_func_ud_39        : dlx_rr_func := "111001";
  constant rr_func_ud_3A        : dlx_rr_func := "111010";
  constant rr_func_ud_3B        : dlx_rr_func := "111011";
  constant rr_func_ud_3C        : dlx_rr_func := "111100";
  constant rr_func_ud_3D        : dlx_rr_func := "111101";
  constant rr_func_ud_3E        : dlx_rr_func := "111110";
  constant rr_func_ud_3F        : dlx_rr_func := "111111";
  
  --
  -- encoding of register-register ALU floating point functions
  --
  constant fp_func_addf		: dlx_fp_func := "00000";
  constant fp_func_subf		: dlx_fp_func := "00001";
  constant fp_func_multf	: dlx_fp_func := "00010";
  constant fp_func_divf		: dlx_fp_func := "00011";
  constant fp_func_addd		: dlx_fp_func := "00100";
  constant fp_func_subd		: dlx_fp_func := "00101";
  constant fp_func_multd	: dlx_fp_func := "00110";
  constant fp_func_divd		: dlx_fp_func := "00111";
  constant fp_func_cvtf2d	: dlx_fp_func := "01000";
  constant fp_func_cvtf2i	: dlx_fp_func := "01001";
  constant fp_func_cvtd2f	: dlx_fp_func := "01010";
  constant fp_func_cvtd2i	: dlx_fp_func := "01011";
  constant fp_func_cvti2f	: dlx_fp_func := "01100";
  constant fp_func_cvti2d	: dlx_fp_func := "01101";
  constant fp_func_ud_0E	: dlx_fp_func := "01110";
  constant fp_func_ud_0F	: dlx_fp_func := "01111";

  constant fp_func_mult	       	: dlx_fp_func := "10000";
  constant fp_func_multu	: dlx_fp_func := "10001";
  constant fp_func_div  	: dlx_fp_func := "10010";
  constant fp_func_divu		: dlx_fp_func := "10011";  
  constant fp_func_eqf		: dlx_fp_func := "10100";
  constant fp_func_nef		: dlx_fp_func := "10101";
  constant fp_func_ltf		: dlx_fp_func := "10110";
  constant fp_func_lef		: dlx_fp_func := "10111";
  constant fp_func_gtf		: dlx_fp_func := "11000";
  constant fp_func_gef		: dlx_fp_func := "11001";
  constant fp_func_eqd		: dlx_fp_func := "11010";
  constant fp_func_ned		: dlx_fp_func := "11011";
  constant fp_func_ltd		: dlx_fp_func := "11100";
  constant fp_func_led		: dlx_fp_func := "11101";
  constant fp_func_gtd		: dlx_fp_func := "11110";
  constant fp_func_ged		: dlx_fp_func := "11111";

--synopsys synthesis_off
  subtype instr_name is string(1 to 7);
  type name_array is array  (natural range <>) of instr_name;
  
  constant opcode_names  : name_array(0 to 63);
  constant rr_func_names : name_array(0 to 63);
  constant fp_func_names : name_array(0 to 31);

  procedure write_instr (L : inout line; instr : in dlx_word);
--synopsys synthesis_on

end dlx_instructions;

