--------------------------------------------------------------------------
--  DLX PROCESSOR MODEL SUITE
--  Copyright (C) 1995, Martin Gumm
--  University of Stuttgart / Department of Computer Science / IPVR-ISE
--------------------------------------------------------------------------
--  This program is free software; you can redistribute it and/or modify
--  it under the terms of the GNU General Public License as published by
--  the Free Software Foundation; either version 1, or (at your option)
--  any later version.
--
--  This program is distributed in the hope that it will be useful,
--  but WITHOUT ANY WARRANTY; without even the implied warranty of
--  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--  GNU General Public License for more details.
--------------------------------------------------------------------------
--  Last revision date : November 15 1995
--------------------------------------------------------------------------

--------------------------------------------------------------------------
--  behavioural architecture of the DLX processor
--  (only signed instructions)
--  
--  (file dlx-behaviour_s.vhd)
--------------------------------------------------------------------------

library IEEE_EXTD;
use IEEE_EXTD.stdl1164_vector_arithmetic.all,
    IEEE_EXTD.stdl1164_extended.all;

use STD.textio.all;

use WORK.dlx_instructions.all;

architecture behaviour_u of dlx is

begin                    -- behaviour
  
  interpreter: process
    --
    -- the following variables correspond to DLXS registers or
    -- bits or address fields
    --
    variable GP_REG : dlx_word_array(reg_index);
    variable PC  : dlx_word;
    variable IR  : dlx_word;
    variable IAR : dlx_word;
    variable ICR : dlx_word;
    variable TBR : dlx_word;    
    variable MAR : dlx_word;
    variable MDR : dlx_word;
    
    alias IR_opcode   : dlx_opcode is IR(0 to 5);
    alias IR_rr_func  : dlx_rr_func is IR(26 to 31);
    alias IR_rs1      : dlx_reg_addr is IR(6 to 10);
    alias IR_rs2      : dlx_reg_addr is IR(11 to 15);
    alias IR_rd_Itype : dlx_reg_addr is IR(11 to 15);
    alias IR_rd_Rtype : dlx_reg_addr is IR(16 to 20);
    alias IR_immed16  : dlx_immed16 is IR(16 to 31);
    alias IR_immed26  : dlx_immed26 is IR(6 to 31);

    alias S    : std_logic is ICR(31);             -- Supervisor bit ('1': supervisor)
    alias IOC  : std_logic is ICR(30);             -- illegal opcode
    alias IRRA : std_logic is ICR(29);             -- illegal reg-reg ALU function 
    alias IAV  : std_logic is ICR(28);             -- instruction address violation
    alias DAV  : std_logic is ICR(27);             -- data address violation
    alias OVAD : std_logic is ICR(26);             -- address calcuation overflow
    alias OVAR : std_logic is ICR(25);             -- arithmetic overflow
    alias PRIV : std_logic is ICR(24);             -- priviledge violation  
    alias MASK : dlx_nibble is ICR(20 to 23);      -- interrupt mask bits
    alias INT  : dlx_nibble is ICR(16 to 19);      -- interrupt bits

    variable rs1, rs2 : reg_index;
    variable rd_Itype : reg_index;
    variable rd_Rtype : reg_index;

    --
    -- further variables
    --
    constant PC_incr : dlx_word := To_StdLogicVector(X"0000_0004");

    variable L : line;
    variable tmp_word : dlx_word;                  -- temporary storage
    variable exc_flag : boolean;                   -- actual exception
    variable prev_exc_flag : boolean;              -- previous exception
    
    -----------------------------------------------------------------
    -- procedure for checking the interrupt lines 
    -----------------------------------------------------------------
    procedure intrpt_check is
    begin
      --
      -- check for active interrupt line(s)
      --
      if phi1 = '1' then
	INT := INT or intrpt;
      end if;
    end intrpt_check;

    -----------------------------------------------------------------
    -- procedure for checking errors and exceptions 
    -----------------------------------------------------------------
    procedure exc_check (to_check : in std_logic;
                         p_exc : inout boolean) is
    begin
      --
      -- check for errors and exceptions
      --
      if to_check = '1' then               -- exception bit set to 1 ?
	p_exc := true;                   -- -> exception
      end if;
    end exc_check;

    -----------------------------------------------------------------
    -- procedure for checking priviledge violations 
    -----------------------------------------------------------------
    procedure priv_check (adr_msb : in std_logic;
                          s_bit : in std_logic;
                          PRIV : inout std_logic;
                          p_exc : inout boolean) is
    begin
      --
      -- check priviledge violation
      --
      if adr_msb = '0' and s_bit = '0' then  -- msb of address = '0' ?
	PRIV := '1';                         --  and not in supervisor mode ?
        p_exc := true;
      end if;
    end priv_check;
    
    -----------------------------------------------------------------
    -- procedure for checking address alignment violations 
    -----------------------------------------------------------------
    procedure adr_check (adr_ls2 : in std_logic_vector(1 downto 0);
                         data_width : mem_access_type;
                         to_set : inout std_logic;
                         p_exc : inout boolean) is
    begin
      case data_width is
	when word =>
	  --
          -- check the last two bits for being zero
	  --
	  if adr_ls2 /= "00" then
	    to_set := '1';                 -- IAV or DAV
	    p_exc := true;               -- -> exception 
	  end if; 
        when half =>
	  --
          -- check the last bit for being zero
	  --
	  if adr_ls2(0) /= '0' then
	    to_set := '1';                 -- IAV or DAV
	    p_exc := true;               -- -> exception 
	  end if;  
	when byte =>
	  --
	  -- do nothing
	  --
	  null;
      end case;
    end adr_check;

    -----------------------------------------------------------------
    -- procedure for writing on the data bus 
    -----------------------------------------------------------------
    procedure bus_write (address : in dlx_address;
 		         data_width : mem_access_type;
                         data    : in dlx_word) is
    begin
      --
      -- place the address on the address bus after the rising edge of phi2
      --
      wait until phi2 = '1';
      if reset = '1' then
 	return;
      end if;      
      a_bus <= address after tpd_behav; 
      --
      -- place data on the data bus after rising edge of phi2 
      --
      wait until phi2 = '1';
      if reset = '1' then
 	return;
      end if;
      d_bus <= data after tpd_behav;
      --
      -- set ENABLE and RW after the rising edge of phi1 & check for interrupts
      --
      wait until phi1 = '1';
      if reset = '1' then 
        return;
      end if;
      intrpt_check;
      rw <= '0' after tpd_behav;
      case data_width is
	when word   => enable <= "1111" after tpd_behav;
	when half   =>
	  case address(1) is
	    when '0' => enable <= "1100" after tpd_behav;
	    when others => enable <= "0011" after tpd_behav;
 	  end case;
	when others =>
 	  case address(1 downto 0) is
	    when "00" => enable <= "1000" after tpd_behav;
	    when "01" => enable <= "0100" after tpd_behav;
	    when "10" => enable <= "0010" after tpd_behav;
	    when others => enable <= "0001" after tpd_behav;
  	  end case;
      end case ;
      --
      -- wait until data accepted
      --
      loop 
        wait until phi2 = '0';
        exit when ready = '1' or reset = '1';
	intrpt_check;
      end loop;
      --
      -- inactivate lines after rising edge of phi1
      --
      wait until phi1 = '1';
      if reset = '1' then 
        return;
      end if;
      intrpt_check;
      d_bus <= (others => 'Z') after tpd_behav;
      enable <= "0000" after tpd_behav;
      rw <= '0' after tpd_behav;
    end bus_write;

    -----------------------------------------------------------------
    -- procedure for reading from the data bus 
    -----------------------------------------------------------------
    procedure bus_read (address : in dlx_address;
                        data_width : in mem_access_type;
			data    : out dlx_word) is
    begin
      --
      -- place the address on the address bus after the rising edge of phi2
      --
      wait until phi2 = '1';
      if reset = '1' then
 	return;
      end if;      
      a_bus <= address after tpd_behav; 
      --
      -- set ENABLE and RW after the rising edge of phi1 & check for interrupts
      --
      wait until phi1 = '1';
      if reset = '1' then 
        return;
      end if;
      intrpt_check;
      rw <= '1' after tpd_behav;
      case data_width is
	when word   => enable <= "1111" after tpd_behav;
	when half   =>
	  case address(1) is
	    when '0' => enable <= "1100" after tpd_behav;
	    when others => enable <= "0011" after tpd_behav;
 	  end case;
	when others =>
 	  case address(1 downto 0) is
	    when "00" => enable <= "1000" after tpd_behav;
	    when "01" => enable <= "0100" after tpd_behav;
	    when "10" => enable <= "0010" after tpd_behav;
	    when others => enable <= "0001" after tpd_behav;
  	  end case;
      end case ;
      --
      -- wait until data ready
      --
      loop 
        wait until phi2 = '0';
        exit when ready = '1' or reset = '1';
        intrpt_check;
      end loop;
      data := To_UX01(d_bus);                 -- register content can't be 'Z'
      --
      -- inactivate lines after rising edge of phi1
      --
      wait until phi1 = '1';
      if reset = '1' then 
        return;
      end if;
      intrpt_check;
      d_bus <= (others => 'Z') after tpd_behav;
      enable <= "0000" after tpd_behav;
      rw <= '0' after tpd_behav;
    end bus_read;
    
    ---------------------------------------------------------------------
    -- procedure for checking the halt signal and freezing the processor 
    ---------------------------------------------------------------------
    procedure check_halt_signal is
    begin
      if halt = '1' then
        if DEBUG then
          write(L, string'(">>> halt detected, freezing processor state..."));
          writeline(output, L);
        end if;
	--
        -- all outputs to high impedance
        --
        a_bus <= (others => 'Z');
        d_bus <= (others => 'Z');
        enable <= "ZZZZ";
        rw <= 'Z';
        --
        -- freeze processsor
        --
        wait until halt = '0';
	
        if DEBUG then
          write(L, string'(">>> halt disactivated, resuming ..."));
          writeline(output, L);
        end if;
      end if;
    end check_halt_signal;
    
  -----------------------------------------------------------------
  -- begin of the interpreter process
  -----------------------------------------------------------------
    
  begin 

    --
    -- normal fetch-decode-execute loop with interrupt handling
    -- (loop is only be exited in case of asynchronous reset or
    --  exceptions during execution)
    --
    fetch_dec_exc: loop 
      --
      -- check for active reset signal and exit the loop in this case 
      --
      exit fetch_dec_exc when reset = '1';
      --
      -- check for active halt signal and freeze the processor in this case 
      --
      check_halt_signal;
      --
      -- fetch next instruction
      --
      if DEBUG then
        write(L, string'(">>> fetching instruction from address: "));
	write(L, PC, X, up);
        writeline(output, L);
      end if;
      
      bus_read(PC, word, IR);
      exit fetch_dec_exc when reset = '1';
      --
      -- test interrupt signals
      --
      if ((MASK(0) AND INT(0)) = '1') or ((MASK(1) AND INT(1)) = '1') or
	 ((MASK(2) AND INT(2)) = '1') or ((MASK(3) AND INT(3)) = '1') then
	--
	-- interrupt detected (no pending interrupt or exception)
	-- (no intrpt. during exception handling orpending intrpt.
   	--  because mask bits are zero after shifting of ICR)
	--
	if DEBUG then
	  write(L, string'(">>> interrupt detected..."));
	  writeline(output, L);
	end if;

	sv_sll(ICR, ICR, "10000");             -- save old ICR / unmask new intrpt.
                                               -- (shift left by 16)
	S := '1';                              -- set supervisor mode (S = ICR(31))
        IAR := PC;                             -- save PC
	PC := TBR;                             -- load interrupt vector from TBR

      else
        --
	-- no interrupt detected => normal decode and execution
        -- increment the PC to point to the following instruction
	--
	if DEBUG then
	  write(L, string'(">>> incrementing PC..."));
	  writeline(output, L);
	end if;

	sv_add(PC, PC_incr, PC, OVAD);
	exc_check(OVAD, exc_flag);       -- address overflow ?
        exit fetch_dec_exc when exc_flag;    
	--
	-- decode the instruction
	--
	if DEBUG then
	  write(L, string'(">>> decoding instruction..."));
	  writeline(output, L);
	  write_instr(L, IR);
	  writeline(output, L);
	end if;

	rs1            := sv_to_natural(IR_rs1);
	rs2            := sv_to_natural(IR_rs2);
	rd_Itype       := sv_to_natural(IR_rd_Itype);
	rd_Rtype       := sv_to_natural(IR_rd_Rtype);
	--
	-- execute
	--
	case IR_opcode is
	  --
	  -- select between all opcodes and execute function
	  --
	  ---------------------------------------------------
	  when op_rr_alu => 
	    --
	    -- register-register alu function (R-type instruction)
	    --
	    case IR_rr_func is
	      ------------------------------------------------------
	      WHEN rr_func_nop =>
		null;
	      ------------------------------------------------------
	      when rr_func_and => 
		GP_REG(rd_Rtype) := GP_REG(rs1) and GP_REG(rs2);
	      ------------------------------------------------------
	      when rr_func_or => 
		GP_REG(rd_Rtype) := GP_REG(rs1) or GP_REG(rs2);
	      ------------------------------------------------------
	      when rr_func_xor => 
		GP_REG(rd_Rtype) := GP_REG(rs1) xor GP_REG(rs2);
	      ------------------------------------------------------
	      when rr_func_add => 
		sv_add(GP_REG(rs1), GP_REG(rs2), GP_REG(rd_Rtype), OVAR);
	        exc_check(OVAR, exc_flag);                 -- arithmetik overflow ?
		exit fetch_dec_exc when exc_flag;
	      ------------------------------------------------------
	      when rr_func_addu => 
		sv_addu(GP_REG(rs1), GP_REG(rs2), GP_REG(rd_Rtype), OVAR);
	        exc_check(OVAR, exc_flag);                -- arithmetik overflow ?
		exit fetch_dec_exc when exc_flag;
	      ------------------------------------------------------
	      when rr_func_sub => 
		sv_sub(GP_REG(rs1), GP_REG(rs2), GP_REG(rd_Rtype), OVAR);
	        exc_check(OVAR, exc_flag);                -- arithmetik overflow ?
	        exit fetch_dec_exc when exc_flag;
	      ------------------------------------------------------
	      when rr_func_subu => 
		sv_subu(GP_REG(rs1), GP_REG(rs2), GP_REG(rd_Rtype), OVAR);
	        exc_check(OVAR, exc_flag);                -- arithmetik overflow ?
	        exit fetch_dec_exc when exc_flag;
	      ------------------------------------------------------
	      when rr_func_sll => 
		sv_sll(GP_REG(rs1), GP_REG(rd_Rtype), GP_REG(rs2)(27 to 31));
	      ------------------------------------------------------
	      when rr_func_srl => 
		sv_srl(GP_REG(rs1), GP_REG(rd_Rtype), GP_REG(rs2)(27 to 31));
	      ------------------------------------------------------
	      when rr_func_sra => 
		sv_sra(GP_REG(rs1), GP_REG(rd_Rtype), GP_REG(rs2)(27 to 31));
	      ------------------------------------------------------
	      when rr_func_seq | rr_func_sequ => 
		if GP_REG(rs1) = GP_REG(rs2) then
		  GP_REG(rd_Rtype) := To_StdLogicVector(X"0000_0001");
		else
		  GP_REG(rd_Rtype) := To_StdLogicVector(X"0000_0000");
		end if;
	      ------------------------------------------------------
	      when rr_func_sne | rr_func_sneu => 
		if GP_REG(rs1) /= GP_REG(rs2) then
		  GP_REG(rd_Rtype) := To_StdLogicVector(X"0000_0001");
		else
		  GP_REG(rd_Rtype) := To_StdLogicVector(X"0000_0000");
		end if;
	      ------------------------------------------------------
	      when rr_func_slt => 
		GP_REG(rd_Rtype) := To_StdLogicVector(X"0000_0000");
		sv_lt(GP_REG(rs1), GP_REG(rs2), GP_REG(rd_Rtype)(31));

	      ------------------------------------------------------
	      when rr_func_sltu => 
		GP_REG(rd_Rtype) := To_StdLogicVector(X"0000_0000");
		sv_ltu(GP_REG(rs1), GP_REG(rs2), GP_REG(rd_Rtype)(31));
	      ------------------------------------------------------
	      when rr_func_sle => 
		GP_REG(rd_Rtype) := To_StdLogicVector(X"0000_0000");
		sv_le(GP_REG(rs1), GP_REG(rs2), GP_REG(rd_Rtype)(31));
	      ------------------------------------------------------
	      when rr_func_sleu => 
		GP_REG(rd_Rtype) := To_StdLogicVector(X"0000_0000");
		sv_leu(GP_REG(rs1), GP_REG(rs2), GP_REG(rd_Rtype)(31));
	      ------------------------------------------------------	      
	      when rr_func_sgt => 
		GP_REG(rd_Rtype) := To_StdLogicVector(X"0000_0000");
		sv_gt(GP_REG(rs1), GP_REG(rs2), GP_REG(rd_Rtype)(31));
	      ------------------------------------------------------	      
	      when rr_func_sgtu => 
		GP_REG(rd_Rtype) := To_StdLogicVector(X"0000_0000");
		sv_gtu(GP_REG(rs1), GP_REG(rs2), GP_REG(rd_Rtype)(31)); 
	      ------------------------------------------------------
	      when rr_func_sge => 
		GP_REG(rd_Rtype) := To_StdLogicVector(X"0000_0000");
		sv_ge(GP_REG(rs1), GP_REG(rs2), GP_REG(rd_Rtype)(31));
	      ------------------------------------------------------
	      when rr_func_sgeu => 
		GP_REG(rd_Rtype) := To_StdLogicVector(X"0000_0000");
		sv_geu(GP_REG(rs1), GP_REG(rs2), GP_REG(rd_Rtype)(31));
	      ------------------------------------------------------
	      when rr_func_movi2s => 
		priv_check('0', S, PRIV, exc_flag);           -- priviledge violation ?
		exit fetch_dec_exc when exc_flag;
		case rd_Rtype is
		  when 1 => ICR := GP_REG(rs1);
		  when 2 => IAR := GP_REG(rs1);
    		  when 4 => TBR := GP_REG(rs1); 
		  when others => null;
		end case;
	      ------------------------------------------------------
	      when rr_func_movs2i => 
		priv_check('0', S, PRIV, exc_flag);           -- priviledge violation ?
		exit fetch_dec_exc when exc_flag;
		case rs1 is
		  when 1 => GP_REG(rd_Rtype) := ICR;
		  when 2 => GP_REG(rd_Rtype) := IAR;
		  when 4 => GP_REG(rd_Rtype) := TBR;
		  when others => null;
		end case;
	      ------------------------------------------------------
	      when rr_func_lb =>
 	        sv_add(GP_REG(rs1), GP_REG(rs2), MAR);
		priv_check(MAR(0), S, PRIV, exc_flag);          -- priviledge violation ?
		exit fetch_dec_exc when exc_flag;
		--
                bus_read(MAR, byte, MDR);
	        exit fetch_dec_exc when reset = '1';
		case MAR(30 to 31) is
	          when "00" =>
		    GP_REG(rd_Rtype) := sv_sext(MDR(0 to 7), 32);
	          when "01" =>
		    GP_REG(rd_Rtype) := sv_sext(MDR(8 to 15), 32);
	          when "10" =>
		    GP_REG(rd_Rtype) := sv_sext(MDR(16 to 23), 32);
	          when "11" =>
		    GP_REG(rd_Rtype) := sv_sext(MDR(24 to 31), 32);
	          when others => null;
	        end case;
	      ------------------------------------------------------
	      when rr_func_lbu =>
 	        sv_add(GP_REG(rs1), GP_REG(rs2), MAR);
		priv_check(MAR(0), S, PRIV, exc_flag);           -- priviledge violation ?
		exit fetch_dec_exc when exc_flag;
		--
	        exit fetch_dec_exc when reset = '1';
	        case MAR(30 to 31) is
	          when "00" =>
		    GP_REG(rd_Rtype) := sv_zext(MDR(0 to 7), 32);
	          when "01" =>
		    GP_REG(rd_Rtype) := sv_zext(MDR(8 to 15), 32);
	          when "10" =>
		    GP_REG(rd_Rtype) := sv_zext(MDR(16 to 23), 32);
	          when "11" =>
		    GP_REG(rd_Rtype) := sv_zext(MDR(24 to 31), 32);
	          when others => null;
	        end case;
	      ------------------------------------------------------
	      when rr_func_lh => 
 	        sv_add(GP_REG(rs1), GP_REG(rs2), MAR);
		priv_check(MAR(0), S, PRIV, exc_flag);           -- priviledge violation ?
                adr_check(MAR(30 to 31), half, DAV, exc_flag);   -- adr. alignment error ? 
		exit fetch_dec_exc when exc_flag;
                --
                bus_read(MAR, half, MDR);
	        exit fetch_dec_exc when reset = '1';
	        if MAR(30) = '0' then
	          GP_REG(rd_Rtype) := sv_sext(MDR(0 to 15), 32);
	        else
	          GP_REG(rd_Rtype) := sv_sext(MDR(16 to 31), 32);
	        end if;
	      ------------------------------------------------------
	      when rr_func_lhu => 
 	        sv_add(GP_REG(rs1), GP_REG(rs2), MAR);
		priv_check(MAR(0), S, PRIV, exc_flag);           -- priviledge violation ?
                adr_check(MAR(30 to 31), half, DAV, exc_flag);   -- adr. alignment error ? 
		exit fetch_dec_exc when exc_flag;
                --
                bus_read(MAR, half, MDR);
	        exit fetch_dec_exc when reset = '1';
	        if MAR(30) = '0' then
	          GP_REG(rd_Rtype) := sv_zext(MDR(0 to 15), 32);
	        else
	          GP_REG(rd_Rtype) := sv_zext(MDR(16 to 31), 32);
	        end if;		
	      ------------------------------------------------------
	      when rr_func_lw => 
	        sv_add(GP_REG(rs1), GP_REG(rs2), MAR);
		priv_check(MAR(0), S, PRIV, exc_flag);                -- priviledge violation ?
                adr_check(MAR(30 to 31), word, DAV, exc_flag);        -- adr. alignment error ? 
		exit fetch_dec_exc when exc_flag;
                --
                bus_read(MAR, word, MDR);
	        exit fetch_dec_exc when reset = '1';
	        GP_REG(rd_Rtype) := MDR;
	      ------------------------------------------------------	  
	      when rr_func_sb => 
	        sv_add(GP_REG(rs1), GP_REG(rs2), MAR);
		priv_check(MAR(0), S, PRIV, exc_flag);                -- priviledge violation ?
		exit fetch_dec_exc when exc_flag;
		--
		MDR := To_StdLogicVector(X"0000_0000");
	        case MAR(30 to 31) is
	          when "00" =>
		    MDR(0 to 7) := GP_REG(rd_Rtype)(24 to 31);
	          when "01" =>
		    MDR(8 to 15) := GP_REG(rd_Rtype)(24 to 31);
	          when "10" =>
		    MDR(16 to 23) := GP_REG(rd_Rtype)(24 to 31);
	          when "11" =>
		    MDR(24 to 31) := GP_REG(rd_Rtype)(24 to 31);
	          when others => null;
	        end case;
                bus_write(MAR, byte, MDR);
	        exit fetch_dec_exc when reset = '1';
	      ------------------------------------------------------	  
	      when rr_func_sh => 
 	        sv_add(GP_REG(rs1), GP_REG(rs2), MAR);
		priv_check(MAR(0), S, PRIV, exc_flag);                -- priviledge violation ?
                adr_check(MAR(30 to 31), half, DAV, exc_flag);        -- adr. alignment error ? 
		exit fetch_dec_exc when exc_flag;
                --
	        MDR := To_StdLogicVector(X"0000_0000");
	        if MAR(30) = '0' then
	          MDR(0 to 15) := GP_REG(rd_Rtype)(16 to 31);
	        else
	          MDR(16 to 31) := GP_REG(rd_Rtype)(16 to 31);
	        end if;
                bus_write(MAR, half, MDR);
	        exit fetch_dec_exc when reset = '1';
	      ------------------------------------------------------	  
	      when rr_func_sw => 
	        sv_add(GP_REG(rs1), GP_REG(rs2), MAR);
		priv_check(MAR(0), S, PRIV, exc_flag);                -- priviledge violation ?
                adr_check(MAR(30 to 31), word, DAV, exc_flag);        -- adr. alignment error ? 
		exit fetch_dec_exc when exc_flag;
                --
	        MDR := GP_REG(rd_Rtype);
                bus_write(MAR, word, MDR);
	        exit fetch_dec_exc when reset = '1';
	      ------------------------------------------------------
	      when others =>
		assert false
		  report "undefined register-register ALU function"
		  severity note;
		  IRRA := '1';
		  exc_flag := true;
	          exit fetch_dec_exc;
	    end case;
	    --
	    -- end of selecting between register-register ALU functions
	    --
	  ------------------------------------------------------ 
	  when op_j  => 
	    sv_add(PC, sv_sext(IR_immed26, 32), tmp_word);       -- calculate new adr.
	    priv_check(tmp_word(0), S, PRIV, exc_flag);          -- priviledge violation ?
            adr_check(tmp_word(30 to 31), word, IAV, exc_flag);  -- adr. alignment error ? 
            exit fetch_dec_exc when exc_flag;
	    PC := tmp_word;                                      -- store valid new adr. in PC
	  ------------------------------------------------------
	  when op_jal => 
	    GP_REG(link_reg) := PC;
	    sv_add(PC, sv_sext(IR_immed26, 32), tmp_word);       -- calculate new adr.
	    priv_check(tmp_word(0), S, PRIV, exc_flag);          -- priviledge violation ?
            adr_check(tmp_word(30 to 31), word, IAV, exc_flag);  -- adr. alignment error ? 
            exit fetch_dec_exc when exc_flag;
	    PC := tmp_word;                                      -- store valid new adr. in PC
	  ------------------------------------------------------
	  when op_jr => 
	    tmp_word := GP_REG(rs1);                             -- load new adr.
	    priv_check(tmp_word(0), S, PRIV, exc_flag);                -- priviledge violation ?
            adr_check(tmp_word(30 to 31), word, IAV, exc_flag);  -- adr. alignment error ? 
            exit fetch_dec_exc when exc_flag;
	    PC := tmp_word;                                      -- store valid new adr. in PC
	  ------------------------------------------------------	  
	  when op_jalr => 
	    GP_REG(link_reg) := PC;
	    tmp_word := GP_REG(rs1);                             -- load new adr.
	    priv_check(tmp_word(0), S, PRIV, exc_flag);                -- priviledge violation ?
            adr_check(tmp_word(30 to 31), word, IAV, exc_flag);  -- adr. alignment error ? 
            exit fetch_dec_exc when exc_flag;
	    PC := tmp_word;                                      -- store valid new adr. in PC
	  ------------------------------------------------------
	  when op_beqz => 
	    if GP_REG(rs1) = To_StdlogicVector(X"0000_0000") then
	      if DEBUG then
		write(L, string'(">>> branch  taken..."));
		writeline(output, L);
	      end if;
  	      --
	      sv_add(PC, sv_sext(IR_immed16, 32), tmp_word);      -- calculate new adr.
	      priv_check(tmp_word(0), S, PRIV, exc_flag);         -- priviledge violation ?
              adr_check(tmp_word(30 to 31), word, IAV, exc_flag); -- adr. alignment error ? 
              exit fetch_dec_exc when exc_flag;
	      PC := tmp_word;                                     -- store valid new adr. in PC 
	    else
	      if DEBUG then
		write(L, string'(">>> no branch  taken..."));
		writeline(output, L);
	      end if;
	    end if;
	  ------------------------------------------------------
	  when op_bnez =>
	    if GP_REG(rs1) /= To_StdlogicVector(X"0000_0000") then
	      if DEBUG then
		write(L, string'(">>> branch taken..."));
		writeline(output, L);
	      end if;
  	      --
	      sv_add(PC, sv_sext(IR_immed16, 32), tmp_word);      -- calculate new adr.
	      priv_check(tmp_word(0), S, PRIV, exc_flag);         -- priviledge violation ?
              adr_check(tmp_word(30 to 31), word, IAV, exc_flag); -- adr. alignment error ? 
              exit fetch_dec_exc when exc_flag;
	      PC := tmp_word;                                     -- store valid new adr. in PC 
	    else
	      if DEBUG then
		write(L, string'(">>> no branch  taken..."));
		writeline(output, L);
	      end if;
	    end if;
	  ------------------------------------------------------
	  when op_rfe => 
	    if DEBUG then
	      write(L, string'(">>> return from interrupt/exception..."));
	      writeline(output, L);
	    end if;

	    priv_check('0', S, PRIV, exc_flag); -- priviledge violation ?
	    exit fetch_dec_exc when exc_flag;

	    sv_srl(ICR, ICR, "10000");          -- restore old ICR (shift right by 16)
	                                        --  clear all exception bits
	    PC := IAR;                          -- restore PC
	    if prev_exc_flag then               -- clear old exception flag if it was set
	      prev_exc_flag := false;
	    end if;
	  ------------------------------------------------------
	  when op_trap =>
	    if DEBUG then
	      write(L, string'(">>> trap detected..."));
	      writeline(output, L);
	    end if;

	    sv_add(TBR, sv_sext(IR_immed16, 32), tmp_word);
            adr_check(tmp_word(30 to 31), word, IAV, exc_flag);  -- adr. alignment error ? 
            exit fetch_dec_exc when exc_flag;
	    --
	    IAR := PC;                                           -- save PC
	    PC := tmp_word;
	    sv_sll(ICR, ICR, "10000");                   -- save old ICR (shift left by 16)
	    S := '1';                                    -- set S bit
	  ------------------------------------------------------
	  when op_and_i => 
	    GP_REG(rd_Itype) := GP_REG(rs1) and sv_zext(IR_immed16, 32);
	  ------------------------------------------------------
	  when op_or_i => 
	    GP_REG(rd_Itype) := GP_REG(rs1) or sv_zext(IR_immed16, 32);
	  ------------------------------------------------------
	  when op_xor_i => 
	    GP_REG(rd_Itype) := GP_REG(rs1) xor sv_zext(IR_immed16, 32);
	  ------------------------------------------------------
	  when op_lhi => 
	    GP_REG(rd_Itype) := IR_immed16 & To_StdLogicVector(X"0000");
	  ------------------------------------------------------
	  when op_add_i =>
	    sv_add(GP_REG(rs1), sv_sext(IR_immed16, 32), GP_REG(rd_Itype), OVAR);
	    exc_check(OVAR, exc_flag);                     -- arithmetik overflow ?
            exit fetch_dec_exc when exc_flag;
	  ------------------------------------------------------
	  when op_addu_i => 
	    sv_addu(GP_REG(rs1), sv_zext(IR_immed16, 32), GP_REG(rd_Itype), OVAR);
	    exc_check(OVAR, exc_flag);                     -- arithmetik overflow ?
            exit fetch_dec_exc when exc_flag;
	  ------------------------------------------------------
	  when op_sub_i => 
	    sv_sub(GP_REG(rs1), sv_sext(IR_immed16, 32), GP_REG(rd_Itype), OVAR);
	    exc_check(OVAR, exc_flag);                     -- arithmetik overflow ?
            exit fetch_dec_exc when exc_flag;
	  ------------------------------------------------------
	  when op_subu_i => 
	    sv_subu(GP_REG(rs1), sv_zext(IR_immed16, 32), GP_REG(rd_Itype), OVAR);
	    exc_check(OVAR, exc_flag);                     -- arithmetik overflow ?
            exit fetch_dec_exc when exc_flag;
	  ------------------------------------------------------
	  when op_sll_i => 
	    sv_sll(GP_REG(rs1), GP_REG(rd_Itype), IR_immed16(11 to 15));
	  ------------------------------------------------------
	  when op_srl_i =>
	    sv_srl(GP_REG(rs1), GP_REG(rd_Itype), IR_immed16(11 to 15));
	  ------------------------------------------------------
	  when op_sra_i => 
	    sv_sra(GP_REG(rs1), GP_REG(rd_Itype), IR_immed16(11 to 15));
	  ------------------------------------------------------
	  when op_seq_i => 
	   if GP_REG(rs1) =  sv_sext(IR_immed16, 32) then
	     GP_REG(rd_Itype) := To_StdLogicVector(X"0000_0001");
	   else
	     GP_REG(rd_Itype) := To_StdLogicVector(X"0000_0000");
	   end if;
	  ------------------------------------------------------
	  when op_sequ_i => 
	   if GP_REG(rs1) =  sv_zext(IR_immed16, 32) then
	     GP_REG(rd_Itype) := To_StdLogicVector(X"0000_0001");
	   else
	     GP_REG(rd_Itype) := To_StdLogicVector(X"0000_0000");
	   end if;
	  ------------------------------------------------------
	  when op_sne_i => 
	   if GP_REG(rs1) /=  sv_sext(IR_immed16, 32) then
	     GP_REG(rd_Itype) := To_StdLogicVector(X"0000_0001");
	   else
	     GP_REG(rd_Itype) := To_StdLogicVector(X"0000_0000");
	   end if;
	  ------------------------------------------------------
	  when op_sneu_i => 
	   if GP_REG(rs1) /=  sv_zext(IR_immed16, 32) then
	     GP_REG(rd_Itype) := To_StdLogicVector(X"0000_0001");
	   else
	     GP_REG(rd_Itype) := To_StdLogicVector(X"0000_0000");
	   end if;
	  ------------------------------------------------------
	  when op_slt_i => 
	    GP_REG(rd_Itype) := To_StdLogicVector(X"0000_0000");
	    sv_lt(GP_REG(rs1), sv_sext(IR_immed16, 32), GP_REG(rd_Itype)(31));
	  ------------------------------------------------------
	  when op_sltu_i => 
	    GP_REG(rd_Itype) := To_StdLogicVector(X"0000_0000");
	    sv_ltu(GP_REG(rs1), sv_zext(IR_immed16, 32), GP_REG(rd_Itype)(31));
	  ------------------------------------------------------
	  when op_sle_i => 
	    GP_REG(rd_Itype) := To_StdLogicVector(X"0000_0000");
	    sv_le(GP_REG(rs1), sv_sext(IR_immed16, 32), GP_REG(rd_Itype)(31));
	  ------------------------------------------------------
	  when op_sleu_i => 
	    GP_REG(rd_Itype) := To_StdLogicVector(X"0000_0000");
	    sv_leu(GP_REG(rs1), sv_zext(IR_immed16, 32), GP_REG(rd_Itype)(31));
	  ------------------------------------------------------	      
	  when op_sgt_i => 
	    GP_REG(rd_Itype) := To_StdLogicVector(X"0000_0000");
	    sv_gt(GP_REG(rs1), sv_sext(IR_immed16, 32), GP_REG(rd_Itype)(31));
	  ------------------------------------------------------
	  when op_sgtu_i => 
	    GP_REG(rd_Itype) := To_StdLogicVector(X"0000_0000");
	    sv_gtu(GP_REG(rs1), sv_zext(IR_immed16, 32), GP_REG(rd_Itype)(31)); 
	  ------------------------------------------------------
	  when op_sge_i => 
	    GP_REG(rd_Itype) := To_StdLogicVector(X"0000_0000");
	    sv_ge(GP_REG(rs1), sv_sext(IR_immed16, 32), GP_REG(rd_Itype)(31));
	  ------------------------------------------------------
	  when op_sgeu_i => 
	    GP_REG(rd_Itype) := To_StdLogicVector(X"0000_0000");
	    sv_geu(GP_REG(rs1), sv_zext(IR_immed16, 32), GP_REG(rd_Itype)(31));
	  ------------------------------------------------------
	  when op_lb_i =>
	    sv_add(GP_REG(rs1), sv_sext(IR_immed16, 32), MAR);
	    priv_check(MAR(0), S, PRIV, exc_flag);        -- priviledge violation ?
	    exit fetch_dec_exc when exc_flag;
	    --
            bus_read(MAR, byte, MDR);
	    exit fetch_dec_exc when reset = '1';
	    case MAR(30 to 31) is
	      when "00" =>
		GP_REG(rd_Itype) := sv_sext(MDR(0 to 7), 32);
	      when "01" =>
		GP_REG(rd_Itype) := sv_sext(MDR(8 to 15), 32);
	      when "10" =>
		GP_REG(rd_Itype) := sv_sext(MDR(16 to 23), 32);
	      when "11" =>
		GP_REG(rd_Itype) := sv_sext(MDR(24 to 31), 32);
	      when others => null;
	    end case;
	  ------------------------------------------------------
	  when op_lbu_i =>
	    sv_add(GP_REG(rs1), sv_sext(IR_immed16, 32), MAR);
	    priv_check(MAR(0), S, PRIV, exc_flag);              -- priviledge violation ?
	    exit fetch_dec_exc when exc_flag;
	    --
            bus_read(MAR, byte, MDR);
	    exit fetch_dec_exc when reset = '1';
	    case MAR(30 to 31) is
	      when "00" =>
		GP_REG(rd_Itype) := sv_zext(MDR(0 to 7), 32);
	      when "01" =>
		GP_REG(rd_Itype) := sv_zext(MDR(8 to 15), 32);
	      when "10" =>
		GP_REG(rd_Itype) := sv_zext(MDR(16 to 23), 32);
	      when "11" =>
		GP_REG(rd_Itype) := sv_zext(MDR(24 to 31), 32);
	      when others => null;
	    end case;
	  ------------------------------------------------------
	  when op_lh_i => 
	    sv_add(GP_REG(rs1), sv_sext(IR_immed16, 32), MAR);
	    priv_check(MAR(0), S, PRIV, exc_flag);                -- priviledge violation ?
            adr_check(MAR(30 to 31), half, DAV, exc_flag);        -- adr. alignment error ? 
	    exit fetch_dec_exc when exc_flag;
            --
            bus_read(MAR, half, MDR);
	    exit fetch_dec_exc when reset = '1';
	    if MAR(30) = '0' then
	      GP_REG(rd_Itype) := sv_sext(MDR(0 to 15), 32);
	    else
	      GP_REG(rd_Itype) := sv_sext(MDR(16 to 31), 32);
	    end if;
	  ------------------------------------------------------
	  when op_lhu_i => 
	    sv_add(GP_REG(rs1), sv_sext(IR_immed16, 32), MAR);
	    priv_check(MAR(0), S, PRIV, exc_flag);               -- priviledge violation ?
            adr_check(MAR(30 to 31), half, DAV, exc_flag);       -- adr. alignment error ? 
	    exit fetch_dec_exc when exc_flag;
            --
            bus_read(MAR, half, MDR);
	    exit fetch_dec_exc when reset = '1';
	    if MAR(30) = '0' then
	      GP_REG(rd_Itype) := sv_zext(MDR(0 to 15), 32);
	    else
	      GP_REG(rd_Itype) := sv_zext(MDR(16 to 31), 32);
	    end if;
	  ------------------------------------------------------
	  when op_lw_i => 
 	    sv_add(GP_REG(rs1), sv_sext(IR_immed16, 32), MAR);
	    priv_check(MAR(0), S, PRIV, exc_flag);                -- priviledge violation ?
            adr_check(MAR(30 to 31), word, DAV, exc_flag);        -- adr. alignment error ? 
	    exit fetch_dec_exc when exc_flag;
            --
            bus_read(MAR, word, MDR);
	    exit fetch_dec_exc when reset = '1';
	    GP_REG(rd_Itype) := MDR;
	  ------------------------------------------------------	  
	  when op_sb_i => 
	    sv_add(GP_REG(rs1), sv_sext(IR_immed16, 32), MAR);
	    priv_check(MAR(0), S, PRIV, exc_flag);                 -- priviledge violation ?
	    exit fetch_dec_exc when exc_flag;
	    --
	    MDR := To_StdLogicVector(X"0000_0000");
	    case MAR(30 to 31) is
	      when "00" =>
		MDR(0 to 7) := GP_REG(rd_Itype)(24 to 31);
	      when "01" =>
		MDR(8 to 15) := GP_REG(rd_Itype)(24 to 31);
	      when "10" =>
		MDR(16 to 23) := GP_REG(rd_Itype)(24 to 31);
	      when "11" =>
		MDR(24 to 31) := GP_REG(rd_Itype)(24 to 31);
	      when others => null;
	    end case;
            bus_write(MAR, byte, MDR);
	    exit fetch_dec_exc when reset = '1';
	  ------------------------------------------------------	  
	  when op_sh_i => 
 	    sv_add(GP_REG(rs1), sv_sext(IR_immed16, 32), MAR);
	    priv_check(MAR(0), S, PRIV, exc_flag);                -- priviledge violation ?
            adr_check(MAR(30 to 31), half, DAV, exc_flag);        -- adr. alignment error ? 
	    exit fetch_dec_exc when exc_flag;
            --
	    MDR := To_StdLogicVector(X"0000_0000");
	    if MAR(30) = '0' then
	      MDR(0 to 15) := GP_REG(rd_Itype)(16 to 31);
	    else
	      MDR(16 to 31) := GP_REG(rd_Itype)(16 to 31);
	    end if;
            bus_write(MAR, half, MDR);
	    exit fetch_dec_exc when reset = '1';
	  ------------------------------------------------------	  
	  when op_sw_i => 
 	    sv_add(GP_REG(rs1), sv_sext(IR_immed16, 32), MAR);
	    priv_check(MAR(0), S, PRIV, exc_flag);                -- priviledge violation ?
            adr_check(MAR(30 to 31), word, DAV, exc_flag);        -- adr. alignment error ? 
	    exit fetch_dec_exc when exc_flag;
            --
	    MDR := GP_REG(rd_Itype);
            bus_write(MAR, word, MDR);
	    exit fetch_dec_exc when reset = '1';
	  ------------------------------------------------------
	  when others =>
	    assert false
	      report "undefined opcode"
	      severity note;
	      IOC := '1';
	      exc_flag := true;
	      exit fetch_dec_exc;   
	  ------------------------------------------------------	  	
	  --
	  -- end of selecting between opcodes
	  --
	end case;
	--
	-- fix up R0 in case it was overwritten
	--
	GP_REG(0) := To_StdLogicVector(X"0000_0000");

	if DEBUG then
	  write(L, string'(">>> end of execution"));
	  writeline(output, L);
	end if;
        --
   	-- end of normal execution (no interrupt)
	--
      end if;
      --
      -- loop is only exited when reset is active or exeception is detected
      --      
    end loop;


    --
    -- check for illegal exiting of fetch_decode_execute loop (assertion)
    --
    assert reset = '1' or exc_flag = true  
      report "DLX: main loop exited unexpectedly"
      severity failure;
      

    if reset = '1' then
      --
      -- reset the processor
      -- (no reset for GP_REG(1 to 31) )
      -- (no reset neccessary for IR, IAR, MAR, MDR)
      --
      if DEBUG then
        write(L, string'(">>> reset detected, resetting processor..."));
        writeline(output, L);
      end if;
	
      d_bus <= (others => 'Z');
      enable <= "0000";
      rw <= '0';
      GP_REG(0) := (others => '0');
      PC  := (others => '0');
      ICR := To_StdLogicVector(X"0000_0001");
      exc_flag := false;
      prev_exc_flag := false;
      --
      -- wait for inactivation of reset signal
      --
      wait until phi2 = '0' and reset = '0';

    elsif exc_flag and prev_exc_flag then
      --
      -- error state
      --
      if DEBUG then
        write(L, string'(">>> error detected, waiting for reset..."));
        writeline(output, L);
      end if;
      --
      -- all outputs to high impedance, error to 1
      --
      a_bus <= (others => 'Z');
      d_bus <= (others => 'Z');
      enable <= "ZZZZ";
      rw <= 'Z';
      error <= '1';
      --
      -- freeze processsor
      --
      wait until reset = '1';

    else
      --
      -- exception handling
      --
      if DEBUG then
        write(L, string'(">>> exception detected..."));
        writeline(output, L);
      end if;

      prev_exc_flag := true;                      -- set flag that exception occurred
      exc_flag := false;                          -- clear actual exception flag
      ICR := ICR(16 to 23) &                      -- save old ICR (shift left by 16) but
             to_stdlogicvector(X"0000") &         --  leave EXC bits unchanged 
 	     ICR(24 to 31);                       --  (exc bits are set after shift by HW)
      S := '1';                                   -- set S bit
      IAR := PC;                                  -- save PC
      PC := TBR;                                  -- load interrupt vector
    end if; 

    --
    -- process interpreter now starts again from beginning
    --
  end process interpreter;

end behaviour_u;


