--------------------------------------------------------------------------
--  DLX PROCESSOR MODEL SUITE
--  Copyright (C) 1995, Martin Gumm
--  University of Stuttgart / Department of Computer Science / IPVR-ISE
--------------------------------------------------------------------------
--  This program is free software; you can redistribute it and/or modify
--  it under the terms of the GNU General Public License as published by
--  the Free Software Foundation; either version 1, or (at your option)
--  any later version.
--
--  This program is distributed in the hope that it will be useful,
--  but WITHOUT ANY WARRANTY; without even the implied warranty of
--  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--  GNU General Public License for more details.
--------------------------------------------------------------------------
--  Last revision date : November 15 1995
--------------------------------------------------------------------------

--------------------------------------------------------------------------
--  behavioural architecture of the DLX processor
--  (third implementation step of the behaviour)
--  
--  (file dlx-behaviour_3.vhd)
--------------------------------------------------------------------------

library IEEE_EXTD;
use IEEE_EXTD.stdl1164_vector_arithmetic.all,
    IEEE_EXTD.stdl1164_extended.all;

use STD.textio.all;

use WORK.dlx_instructions.all,
    WORK.behav_procedures_123.all;
  
architecture behaviour_3 of dlx is

begin                    -- behaviour
  
  interpreter: process
    --
    -- the following variables correspond to DLXS registers or
    -- bits or address fields
    --
    variable GP_REG : dlx_word_array(reg_index);
    variable PC  : dlx_word;
    variable IR  : dlx_word;
    variable IAR : dlx_word;
    variable ICR : dlx_word;
    variable TBR : dlx_word;    
    variable MAR : dlx_word;
    variable MDR : dlx_word;
    
    alias IR_opcode   : dlx_opcode is IR(0 to 5);
    alias IR_rr_func  : dlx_rr_func is IR(26 to 31);
    alias IR_rs1      : dlx_reg_addr is IR(6 to 10);
    alias IR_rs2      : dlx_reg_addr is IR(11 to 15);
    alias IR_rd_Itype : dlx_reg_addr is IR(11 to 15);
    alias IR_rd_Rtype : dlx_reg_addr is IR(16 to 20);
    alias IR_immed16  : dlx_immed16 is IR(16 to 31);
    alias IR_immed26  : dlx_immed26 is IR(6 to 31);

    variable rs1, rs2 : reg_index;
    variable rd_Itype : reg_index;
    variable rd_Rtype : reg_index;

    --
    -- further variables
    --
    constant PC_incr : dlx_word := To_StdLogicVector(X"0000_0004");

    variable L : line;
    variable tmp_word : dlx_word;                  -- temporary storage

  -----------------------------------------------------------------
  -- begin of the interpreter process
  -----------------------------------------------------------------
    
  begin 
    --
    -- normal fetch-decode-execute loop with interrupt handling
    -- (loop is only be exited in case of asynchronous reset or
    --  exceptions during execution)
    --
    fetch_dec_exc: loop 
      --
      -- check for active reset signal and exit the loop in this case 
      --
      exit fetch_dec_exc when reset = '1';
      --
      -- fetch next instruction
      --
      if DEBUG then
        write(L, string'(">>> fetching instruction from address: "));
	write(L, PC, X, up);
        writeline(output, L);
      end if;
      
      bus_read(PC, word, IR, phi1, phi2, a_bus, d_bus,
               reset, ready, rw, enable, tpd_behav);
      exit fetch_dec_exc when reset = '1';
      --
      -- no interrupt detected => normal decode and execution
      -- increment the PC to point to the following instruction
      --
      if DEBUG then
	write(L, string'(">>> incrementing PC..."));
	writeline(output, L);
      end if;

      sv_add(PC, PC_incr, tmp_word);  -- SYNOPSYS 3.2 -> 3.3 FEHLER
      PC := tmp_word;
      --
      -- decode the instruction
      --
      if DEBUG then
	write(L, string'(">>> decoding instruction..."));
	writeline(output, L);
	write_instr(L, IR);
	writeline(output, L);
      end if;

      rs1            := sv_to_natural(IR_rs1);
      rs2            := sv_to_natural(IR_rs2);
      rd_Itype       := sv_to_natural(IR_rd_Itype);
      rd_Rtype       := sv_to_natural(IR_rd_Rtype);
      --
      -- execute
      --
      case IR_opcode is
	--
	-- select between all opcodes and execute function
	--
	---------------------------------------------------
	when op_rr_alu => 
	  --
	  -- register-register alu function (R-type instruction)
	  --
	  case IR_rr_func is
	    ------------------------------------------------------
	    WHEN rr_func_nop =>
	      null;
	    ------------------------------------------------------
	    when rr_func_and => 
	      GP_REG(rd_Rtype) := GP_REG(rs1) and GP_REG(rs2);
	    ------------------------------------------------------
	    when rr_func_or => 
	      GP_REG(rd_Rtype) := GP_REG(rs1) or GP_REG(rs2);
	    ------------------------------------------------------
	    when rr_func_xor => 
	      GP_REG(rd_Rtype) := GP_REG(rs1) xor GP_REG(rs2);
	    ------------------------------------------------------
	    when rr_func_add => 
	      sv_add(GP_REG(rs1), GP_REG(rs2), tmp_word);  -- SYNOPSYS 3.2 -> 3.3 FEHLER
              GP_REG(rd_Rtype) := tmp_word;
	    ------------------------------------------------------
	    when rr_func_sub => 
	      sv_sub(GP_REG(rs1), GP_REG(rs2), tmp_word);  -- SYNOPSYS 3.2 -> 3.3 FEHLER
              GP_REG(rd_Rtype) := tmp_word;
	    ------------------------------------------------------
	    when rr_func_sll => 
	      sv_sll(GP_REG(rs1), GP_REG(rd_Rtype), GP_REG(rs2)(27 to 31));
	    ------------------------------------------------------
	    when rr_func_srl => 
	      sv_srl(GP_REG(rs1), GP_REG(rd_Rtype), GP_REG(rs2)(27 to 31));
	    ------------------------------------------------------
	    when rr_func_sra => 
	      sv_sra(GP_REG(rs1), GP_REG(rd_Rtype), GP_REG(rs2)(27 to 31));
	    -----------------------------------------------------
	    when rr_func_seq => 
	      if GP_REG(rs1) = GP_REG(rs2) then
		GP_REG(rd_Rtype) := To_StdLogicVector(X"0000_0001");
	      else
		GP_REG(rd_Rtype) := To_StdLogicVector(X"0000_0000");
	      end if;
	    ------------------------------------------------------
	    when rr_func_sne => 
	      if GP_REG(rs1) /= GP_REG(rs2) then
		GP_REG(rd_Rtype) := To_StdLogicVector(X"0000_0001");
	      else
		GP_REG(rd_Rtype) := To_StdLogicVector(X"0000_0000");
	      end if;
	    ------------------------------------------------------
	    when rr_func_slt => 
	      GP_REG(rd_Rtype) := To_StdLogicVector(X"0000_0000");
	      sv_lt(GP_REG(rs1), GP_REG(rs2), GP_REG(rd_Rtype)(31));
	    ------------------------------------------------------
	    when rr_func_sle => 
	      GP_REG(rd_Rtype) := To_StdLogicVector(X"0000_0000");
	      sv_le(GP_REG(rs1), GP_REG(rs2), GP_REG(rd_Rtype)(31));
	    ------------------------------------------------------	      
	    when rr_func_sgt => 
	      GP_REG(rd_Rtype) := To_StdLogicVector(X"0000_0000");
	      sv_gt(GP_REG(rs1), GP_REG(rs2), GP_REG(rd_Rtype)(31));
	    ------------------------------------------------------
	    when rr_func_sge => 
	      GP_REG(rd_Rtype) := To_StdLogicVector(X"0000_0000");
	      sv_ge(GP_REG(rs1), GP_REG(rs2), GP_REG(rd_Rtype)(31));
	    ------------------------------------------------------
	    when rr_func_movi2s => 
	      case rd_Rtype is
        	when 1 => ICR := GP_REG(rs1);
		when 2 => IAR := GP_REG(rs1);
		when 4 => TBR := GP_REG(rs1);
		when others => null;
	      end case;
	    ------------------------------------------------------
	    when rr_func_movs2i => 
	      case rs1 is
		when 1 => GP_REG(rd_Rtype) := ICR;
	  	when 2 => GP_REG(rd_Rtype) := IAR;
	  	when 4 => GP_REG(rd_Rtype) := TBR;
	  	when others => null;
	      end case;
	    ------------------------------------------------------
	    when rr_func_lb =>
 	      sv_add(GP_REG(rs1), GP_REG(rs2), MAR);
              bus_read(MAR, byte, MDR,
 	                phi1, phi2, a_bus, d_bus, reset, ready, rw, enable, tpd_behav);
	      exit fetch_dec_exc when reset = '1';
	      case MAR(30 to 31) is
	        when "00" =>
		  GP_REG(rd_Rtype) := sv_sext(MDR(0 to 7), 32);
	        when "01" =>
		  GP_REG(rd_Rtype) := sv_sext(MDR(8 to 15), 32);
	        when "10" =>
		  GP_REG(rd_Rtype) := sv_sext(MDR(16 to 23), 32);
	        when "11" =>
		  GP_REG(rd_Rtype) := sv_sext(MDR(24 to 31), 32);
	        when others => null;
	      end case;
	    ------------------------------------------------------
	    when rr_func_lh => 
 	      sv_add(GP_REG(rs1), GP_REG(rs2), MAR);
              bus_read(MAR, half, MDR,
 	                phi1, phi2, a_bus, d_bus, reset, ready, rw, enable, tpd_behav);
	      exit fetch_dec_exc when reset = '1';
	      if MAR(30) = '0' then
	        GP_REG(rd_Rtype) := sv_sext(MDR(0 to 15), 32);
	      else
	        GP_REG(rd_Rtype) := sv_sext(MDR(16 to 31), 32);
	      end if;
	    ------------------------------------------------------
	    when rr_func_lw => 
	      sv_add(GP_REG(rs1), GP_REG(rs2), MAR);
              bus_read(MAR, word, MDR,
 	                phi1, phi2, a_bus, d_bus, reset, ready, rw, enable, tpd_behav);
	      exit fetch_dec_exc when reset = '1';
	      GP_REG(rd_Rtype) := MDR;
	    ------------------------------------------------------	  
	    when rr_func_sb => 
	      sv_add(GP_REG(rs1), GP_REG(rs2), MAR);
	      MDR := To_StdLogicVector(X"0000_0000");
	      case MAR(30 to 31) is
	        when "00" =>
		  MDR(0 to 7) := GP_REG(rd_Rtype)(24 to 31);
	        when "01" =>
		  MDR(8 to 15) := GP_REG(rd_Rtype)(24 to 31);
	        when "10" =>
		  MDR(16 to 23) := GP_REG(rd_Rtype)(24 to 31);
	        when "11" =>
		  MDR(24 to 31) := GP_REG(rd_Rtype)(24 to 31);
	        when others => null;
	      end case;
              bus_write(MAR, byte, MDR,
 	                phi1, phi2, a_bus, d_bus, reset, ready, rw, enable, tpd_behav);
	      exit fetch_dec_exc when reset = '1';
	    ------------------------------------------------------	  
	    when rr_func_sh => 
 	      sv_add(GP_REG(rs1), GP_REG(rs2), MAR);
	      MDR := To_StdLogicVector(X"0000_0000");
	      if MAR(30) = '0' then
	        MDR(0 to 15) := GP_REG(rd_Rtype)(16 to 31);
	      else
	        MDR(16 to 31) := GP_REG(rd_Rtype)(16 to 31);
	      end if;
              bus_write(MAR, half, MDR,
 	                phi1, phi2, a_bus, d_bus, reset, ready, rw, enable, tpd_behav);
	      exit fetch_dec_exc when reset = '1';
	    ------------------------------------------------------	  
	    when rr_func_sw => 
 	      sv_add(GP_REG(rs1), sv_sext(IR_immed16, 32), MAR);
	      MDR := GP_REG(rd_Rtype);
              bus_write(MAR, word, MDR,
 	                phi1, phi2, a_bus, d_bus, reset, ready, rw, enable, tpd_behav);
 	      exit fetch_dec_exc when reset = '1';
	    ------------------------------------------------------
	    when others =>
	      assert false
		report "undefined register-register ALU function"
		severity note;
	  end case;
	  --
	  -- end of selecting between register-register ALU functions
	  --
	------------------------------------------------------
	when op_j  => 
	  sv_add(PC, sv_sext(IR_immed26, 32),  tmp_word);  -- SYNOPSYS 3.2 -> 3.3 FEHLER
          PC := tmp_word;
	------------------------------------------------------
	when op_jal => 
	  GP_REG(link_reg) := PC;
	  sv_add(PC, sv_sext(IR_immed26, 32), tmp_word);  -- SYNOPSYS 3.2 -> 3.3 FEHLER
          PC := tmp_word;
	------------------------------------------------------
	when op_jr => 
	  PC := GP_REG(rs1);
	------------------------------------------------------	  
	when op_jalr => 
	  GP_REG(link_reg) := PC;
	  PC := GP_REG(rs1);
	------------------------------------------------------
	when op_beqz => 
	  if GP_REG(rs1) = To_StdlogicVector(X"0000_0000") then
	    if DEBUG then
	      write(L, string'(">>> branch  taken..."));
	      writeline(output, L);
	    end if;

	    sv_add(PC, sv_sext(IR_immed16, 32),  tmp_word);  -- SYNOPSYS 3.2 -> 3.3 FEHLER
            PC := tmp_word;
	  else
	    if DEBUG then
	      write(L, string'(">>> no branch  taken..."));
	      writeline(output, L);
	    end if;
	  end if;
	------------------------------------------------------
	when op_bnez =>
	  if GP_REG(rs1) /= To_StdlogicVector(X"0000_0000") then
	    if DEBUG then
	      write(L, string'(">>> branch taken..."));
	      writeline(output, L);
	    end if;

	    sv_add(PC, sv_sext(IR_immed16, 32),  tmp_word);  -- SYNOPSYS 3.2 -> 3.3 FEHLER
            PC := tmp_word;
	  else
	    if DEBUG then
	      write(L, string'(">>> no branch  taken..."));
	      writeline(output, L);
	    end if;
	  end if;
	------------------------------------------------------
	when op_and_i => 
	  GP_REG(rd_Itype) := GP_REG(rs1) and sv_zext(IR_immed16, 32);
	------------------------------------------------------
	when op_or_i => 
	  GP_REG(rd_Itype) := GP_REG(rs1) or sv_zext(IR_immed16, 32);
	------------------------------------------------------
	when op_xor_i => 
	  GP_REG(rd_Itype) := GP_REG(rs1) xor sv_zext(IR_immed16, 32);
	------------------------------------------------------
	when op_lhi => 
	  GP_REG(rd_Itype) := IR_immed16 & To_StdLogicVector(X"0000");
	------------------------------------------------------
	when op_add_i =>
	  sv_add(GP_REG(rs1), sv_sext(IR_immed16, 32), tmp_word);  -- SYNOPSYS 3.2 -> 3.3 FEHLER
          GP_REG(rd_Itype) := tmp_word;
	------------------------------------------------------
	when op_sub_i => 
	  sv_sub(GP_REG(rs1), sv_sext(IR_immed16, 32), tmp_word);  -- SYNOPSYS 3.2 -> 3.3 FEHLER
          GP_REG(rd_Itype) := tmp_word;
	------------------------------------------------------
	when op_sll_i => 
	  sv_sll(GP_REG(rs1), GP_REG(rd_Itype), IR_immed16(11 to 15));
	------------------------------------------------------
	when op_srl_i =>
	  sv_srl(GP_REG(rs1), GP_REG(rd_Itype), IR_immed16(11 to 15));
	------------------------------------------------------
	when op_sra_i => 
	  sv_sra(GP_REG(rs1), GP_REG(rd_Itype), IR_immed16(11 to 15));
	------------------------------------------------------
	when op_seq_i => 
	 if GP_REG(rs1) =  sv_sext(IR_immed16, 32) then
	   GP_REG(rd_Itype) := To_StdLogicVector(X"0000_0001");
	 else
	   GP_REG(rd_Itype) := To_StdLogicVector(X"0000_0000");
	 end if;
	------------------------------------------------------
	when op_sne_i => 
	 if GP_REG(rs1) /=  sv_sext(IR_immed16, 32) then
	   GP_REG(rd_Itype) := To_StdLogicVector(X"0000_0001");
	 else
	   GP_REG(rd_Itype) := To_StdLogicVector(X"0000_0000");
	 end if;
	------------------------------------------------------
	when op_slt_i => 
	  GP_REG(rd_Itype) := To_StdLogicVector(X"0000_0000");
	  sv_lt(GP_REG(rs1), sv_sext(IR_immed16, 32), GP_REG(rd_Itype)(31));
	------------------------------------------------------
	when op_sle_i => 
	  GP_REG(rd_Itype) := To_StdLogicVector(X"0000_0000");
	  sv_le(GP_REG(rs1), sv_sext(IR_immed16, 32), GP_REG(rd_Itype)(31));
	------------------------------------------------------	      
	when op_sgt_i => 
	  GP_REG(rd_Itype) := To_StdLogicVector(X"0000_0000");
	  sv_gt(GP_REG(rs1), sv_sext(IR_immed16, 32), GP_REG(rd_Itype)(31));
	------------------------------------------------------
	when op_sge_i => 
	  GP_REG(rd_Itype) := To_StdLogicVector(X"0000_0000");
	  sv_ge(GP_REG(rs1), sv_sext(IR_immed16, 32), GP_REG(rd_Itype)(31));
	------------------------------------------------------
	when op_lb_i => 
	  sv_add(GP_REG(rs1), sv_sext(IR_immed16, 32), MAR);
          bus_read(MAR, byte, MDR,
 	           phi1, phi2, a_bus, d_bus, reset, ready, rw, enable, tpd_behav);
	  exit fetch_dec_exc when reset = '1';
	  case MAR(30 to 31) is
	    when "00" =>
	      GP_REG(rd_Itype) := sv_sext(MDR(0 to 7), 32);
	    when "01" =>
	      GP_REG(rd_Itype) := sv_sext(MDR(8 to 15), 32);
	    when "10" =>
	      GP_REG(rd_Itype) := sv_sext(MDR(16 to 23), 32);
	    when "11" =>
	      GP_REG(rd_Itype) := sv_sext(MDR(24 to 31), 32);
	    when others => null;
	  end case;
	------------------------------------------------------
	when op_lh_i => 
	  sv_add(GP_REG(rs1), sv_sext(IR_immed16, 32), MAR);
          bus_read(MAR, half, MDR,
 	           phi1, phi2, a_bus, d_bus, reset, ready, rw, enable, tpd_behav);
	  exit fetch_dec_exc when reset = '1';
	  if MAR(30) = '0' then
	    GP_REG(rd_Itype) := sv_sext(MDR(0 to 15), 32);
	  else
	    GP_REG(rd_Itype) := sv_sext(MDR(16 to 31), 32);
	  end if;
	------------------------------------------------------
	when op_lw_i => 
	  sv_add(GP_REG(rs1), sv_sext(IR_immed16, 32), MAR);
          bus_read(MAR, word, MDR,
 	           phi1, phi2, a_bus, d_bus, reset, ready, rw, enable, tpd_behav);
	  exit fetch_dec_exc when reset = '1';
	  GP_REG(rd_Itype) := MDR;
	------------------------------------------------------	  
	when op_sb_i => 
	  sv_add(GP_REG(rs1), sv_sext(IR_immed16, 32), MAR);
	  MDR := To_StdLogicVector(X"0000_0000");
	  case MAR(30 to 31) is
	    when "00" =>
	      MDR(0 to 7) := GP_REG(rd_Itype)(24 to 31);
	    when "01" =>
	      MDR(8 to 15) := GP_REG(rd_Itype)(24 to 31);
	    when "10" =>
	      MDR(16 to 23) := GP_REG(rd_Itype)(24 to 31);
	    when "11" =>
	      MDR(24 to 31) := GP_REG(rd_Itype)(24 to 31);
	    when others => null;
	  end case;
          bus_write(MAR, byte, MDR,
 	            phi1, phi2, a_bus, d_bus, reset, ready, rw, enable, tpd_behav);
	  exit fetch_dec_exc when reset = '1';
	------------------------------------------------------	  
	when op_sh_i => 
	  sv_add(GP_REG(rs1), sv_sext(IR_immed16, 32), MAR);
	  MDR := To_StdLogicVector(X"0000_0000");
	  if MAR(30) = '0' then
	    MDR(0 to 15) := GP_REG(rd_Itype)(16 to 31);
	  else
	    MDR(16 to 31) := GP_REG(rd_Itype)(16 to 31);
	  end if;
          bus_write(MAR, half, MDR,
 	            phi1, phi2, a_bus, d_bus, reset, ready, rw, enable, tpd_behav);
	  exit fetch_dec_exc when reset = '1';
	------------------------------------------------------	  
	when op_sw_i => 
	  sv_add(GP_REG(rs1), sv_sext(IR_immed16, 32), MAR);
	  MDR := GP_REG(rd_Itype);
          bus_write(MAR, word, MDR,
 	            phi1, phi2, a_bus, d_bus, reset, ready, rw, enable, tpd_behav);
	  exit fetch_dec_exc when reset = '1';
	------------------------------------------------------
	when others =>
	  assert false
	    report "undefined opcode"
	    severity note;
	------------------------------------------------------	  	
	--
	-- end of selecting between opcodes
	--
      end case;
      --
      -- fix up R0 in case it was overwritten
      --
      GP_REG(0) := To_StdLogicVector(X"0000_0000");
      --
      -- leave loop when exception detected
      --
      if DEBUG then
	write(L, string'(">>> end of execution"));
	writeline(output, L);
      end if;
      --
      -- loop is only exited when reset is active
      --      
    end loop;

    --
    -- check for illegal exiting of fetch_decode_execute loop (assertion)
    --
    assert reset = '1'
      report "DLX: main loop exited unexpectedly"
      severity failure;
      
    if reset = '1' then
      --
      -- reset the processor<check for
      -- (no reset for GP_REG(1 to 31) )
      -- (no reset neccessary for IR, IAR, MAR, MDR)
      --
      if DEBUG then
        write(L, string'(">>> reset detected, resetting processor..."));
        writeline(output, L);
      end if;
	
      d_bus <= (others => 'Z');
      enable <= "0000";
      rw <= '0';
      GP_REG(0) := (others => '0');
      PC  := (others => '0');
      ICR := To_StdLogicVector(X"0000_0001");
      --
      -- wait for inactivation of reset signal
      --
      wait until phi2 = '0' and reset = '0';
    end if; 

    --
    -- process interpreter now starts again from beginning
    --
  end process interpreter;

end behaviour_3;


